<?php
/**
 * Upload helper.
 *
 * Supported signatures:
 * 1) Legacy: save_upload($_FILES['field'], $userId, $subDir='attachments') -> returns attachment_id (int)
 * 2) Simple: save_upload('fieldName', 'subDir') -> returns public URL (string)
 */

require_once __DIR__ . '/db.php';

if (!function_exists('save_upload')) {
    /**
     * @param array|string $fileOrField
     * @param int|string   $userOrSubDir
     * @param string       $subDir
     * @return int|string|null
     */
    function save_upload($fileOrField, $userOrSubDir = 'attachments', string $subDir = 'attachments') {
        $cfg = require __DIR__ . '/config.php';

        $isLegacy = is_array($fileOrField);
        if ($isLegacy) {
            $file = $fileOrField;
            $uploadedBy = is_numeric($userOrSubDir) ? (int)$userOrSubDir : null;
            $targetSubDir = $subDir;
        } else {
            $fieldName = (string)$fileOrField;
            if (empty($_FILES[$fieldName]['tmp_name'])) return null;
            $file = $_FILES[$fieldName];
            $uploadedBy = null;
            $targetSubDir = (string)$userOrSubDir;
        }

        if (empty($file['tmp_name']) || ($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) return null;

        $maxBytes = (int)($cfg['uploads']['max_bytes'] ?? (8 * 1024 * 1024));
        if (!empty($file['size']) && (int)$file['size'] > $maxBytes) return null;

        $allowed = $cfg['uploads']['allowed_mimes'] ?? ['image/jpeg','image/png','application/pdf'];
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        if (!in_array($mime, $allowed, true)) return null;

        $extMap = [
            'image/jpeg' => 'jpg',
            'image/png'  => 'png',
            'image/gif'  => 'gif',
            'image/webp' => 'webp',
            'application/pdf' => 'pdf',
        ];
        $ext = $extMap[$mime] ?? strtolower(pathinfo($file['name'] ?? '', PATHINFO_EXTENSION));
        if ($ext === '') $ext = 'bin';

        $uploadsRoot = $cfg['uploads']['dir'] ?? (__DIR__ . '/../uploads');
        if (!is_dir($uploadsRoot)) mkdir($uploadsRoot, 0755, true);

        $targetSubDir = trim($targetSubDir, '/');
        if ($targetSubDir === '') $targetSubDir = 'attachments';
        $dir = rtrim($uploadsRoot, '/\\') . DIRECTORY_SEPARATOR . $targetSubDir;
        if (!is_dir($dir)) mkdir($dir, 0755, true);

        $fname = date('Ymd_His') . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
        $dest = $dir . DIRECTORY_SEPARATOR . $fname;
        if (!move_uploaded_file($file['tmp_name'], $dest)) return null;

        $baseUrl = rtrim((string)($cfg['app']['base_url'] ?? ''), '/');
        $publicUrl = $baseUrl . '/uploads/' . $targetSubDir . '/' . $fname;

        if ($isLegacy) {
            $pdo = db();
            $st = $pdo->prepare("INSERT INTO attachments (filename, path, mime, size_bytes, uploaded_by, original_name) VALUES (?,?,?,?,?,?)");
            $st->execute([
                $fname,
                $publicUrl,
                $mime,
                (int)($file['size'] ?? 0),
                $uploadedBy,
                $file['name'] ?? null,
            ]);
            return (int)$pdo->lastInsertId();
        }

        return $publicUrl;
    }
}
