<?php
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/docno.php';
require_once __DIR__ . '/../app/upload.php';

$me  = require_role(['ADMIN','WAREHOUSE']);
$pdo = db();

/* ========== دوال مساعدة ========== */

function fetch_distributors(PDO $pdo): array {
  return $pdo->query("SELECT id, name FROM distributors WHERE active=1 ORDER BY name")->fetchAll();
}

function fetch_issues_with_stock(PDO $pdo): array {
  $st = $pdo->query("
    SELECT
      i.id,
      p.name                                 AS product_name,
      i.issue_month,
      i.issue_year,
      i.issue_number,
      i.title,
      (
        COALESCE((
            SELECT SUM(wri.qty)
            FROM warehouse_receipt_items wri
            JOIN warehouse_receipts wr ON wr.id = wri.receipt_id
            WHERE wri.issue_id = i.id AND wr.status = 'APPROVED'
        ),0)
        -
        COALESCE((
            SELECT SUM(ddi.qty)
            FROM distributor_dispatch_items ddi
            JOIN distributor_dispatches dd ON dd.id = ddi.dispatch_id
            WHERE ddi.issue_id = i.id AND dd.status = 'APPROVED'
        ),0)
        +
        COALESCE((
            SELECT SUM(wrti.qty)
            FROM warehouse_return_items wrti
            JOIN warehouse_returns wrt ON wrt.id = wrti.return_id
            WHERE wrti.issue_id = i.id AND wrt.status = 'APPROVED'
        ),0)
      ) AS available_qty
    FROM issues i
    JOIN products p ON p.id = i.product_id
    WHERE i.status = 'OPEN'
    ORDER BY i.issue_year DESC, i.issue_month DESC, p.name
  ");
  return $st->fetchAll();
}

function dis_status_ar(string $s): array {
  return match($s) {
    'DRAFT'     => ['مسودة',  'badge-draft'],
    'SUBMITTED' => ['مرسلة',  'badge-submitted'],
    'APPROVED'  => ['معتمدة', 'badge-approved'],
    'REJECTED'  => ['مرفوضة','badge-rejected'],
    default     => [$s,        'badge-draft'],
  };
}

/* ========== متغيرات عامة ========== */

$error_message = null;
$csrf          = csrf_token();

/* ========== معالجة POST ========== */

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $op = $_POST['op'] ?? '';

  /* إنشاء واعتماد أمر صرف مباشر من المودال */
  if ($op === 'create_and_approve') {
    $distributor_id = (int)($_POST['distributor_id'] ?? 0);
    $issue_id       = (int)($_POST['issue_id'] ?? 0);
    $qty            = (int)($_POST['qty'] ?? 0);

    if ($distributor_id<=0 || $issue_id<=0 || $qty<=0) {
      $error_message = "برجاء اختيار الموزع والإصدار وإدخال كمية صحيحة.";
    } else {
      // التحقق من إيقاف الموزع
      $st = $pdo->prepare("SELECT suspended, suspended_reason FROM distributors WHERE id = ?");
      $st->execute([$distributor_id]);
      $dist = $st->fetch();

      if ($dist && $dist['suspended']) {
        $error_message = "لا يمكن صرف منتجات لهذا الموزع لأنه موقوف من الإدارة. السبب: " . htmlspecialchars($dist['suspended_reason']);
      } else {
        // حساب الرصيد المتاح للإصدار
        $st = $pdo->prepare("
            SELECT
                COALESCE((
                    SELECT SUM(wri.qty)
                    FROM warehouse_receipt_items wri
                    JOIN warehouse_receipts wr ON wr.id = wri.receipt_id
                    WHERE wri.issue_id = ? AND wr.status = 'APPROVED'
                ),0)
                -
                COALESCE((
                    SELECT SUM(ddi.qty)
                    FROM distributor_dispatch_items ddi
                    JOIN distributor_dispatches dd ON dd.id = ddi.dispatch_id
                    WHERE ddi.issue_id = ? AND dd.status = 'APPROVED'
                ),0)
                +
                COALESCE((
                    SELECT SUM(wrti.qty)
                    FROM warehouse_return_items wrti
                    JOIN warehouse_returns wrt ON wrt.id = wrti.return_id
                    WHERE wrti.issue_id = ? AND wrt.status = 'APPROVED'
                ),0)
                AS available_qty
        ");
        $st->execute([$issue_id, $issue_id, $issue_id]);
        $available = (int)$st->fetchColumn();

        if ($qty > $available) {
          $error_message = "لا يمكن صرف هذه الكمية. المتاح بالمخزن من هذا الإصدار هو: " . $available;
        } else {
          // 1) إنشاء أمر صرف بحالة APPROVED
          $st = $pdo->prepare("
            INSERT INTO distributor_dispatches (dispatch_no, distributor_id, dispatch_at, created_by, status)
            VALUES (NULL, ?, NOW(), ?, 'APPROVED')
          ");
          $st->execute([$distributor_id, (int)$me['id']]);
          $dispatch_id = (int)$pdo->lastInsertId();

          // 2) توليد رقم المستند
          $no = next_doc_no($pdo, 'DIS', date('Y-m-d H:i:s'));
          $pdo->prepare("UPDATE distributor_dispatches SET dispatch_no=? WHERE id=?")
              ->execute([$no, $dispatch_id]);

          // 3) إضافة الصنف
          $st = $pdo->prepare("
            INSERT INTO distributor_dispatch_items (dispatch_id, issue_id, qty)
            VALUES (?,?,?)
          ");
          $st->execute([$dispatch_id, $issue_id, $qty]);

          // 4) رفع POD إن وجد
          if (!empty($_FILES['pod']) && is_array($_FILES['pod']) && $_FILES['pod']['error'] === UPLOAD_ERR_OK) {
            $attId = save_upload($_FILES['pod'], (int)$me['id']);
            $pdo->prepare("UPDATE distributor_dispatches SET pod_attachment_id=? WHERE id=?")
                ->execute([$attId, $dispatch_id]);
          }

          // 5) الرجوع لجدول آخر 200 أمر (لا توجد مسودات)
          header("Location: dispatches.php");
          exit;
        }
      }
    }
  }
}

/* ========== بيانات الجدول والقوائم ========== */

$rows = $pdo->query("
  SELECT
    d.id,
    d.dispatch_no,
    d.dispatch_at,
    d.status,
    dist.name AS distributor_name,
    u.name   AS created_by_name,
    GROUP_CONCAT(
      CONCAT(
        p.name, ' - ',
        LPAD(i.issue_month,2,'0'), '/', i.issue_year,
        ' (', ddi.qty, ')'
      )
      ORDER BY i.issue_year DESC, i.issue_month DESC, p.name
      SEPARATOR ' | '
    ) AS items_summary
  FROM distributor_dispatches d
  JOIN distributors dist ON dist.id = d.distributor_id
  JOIN users       u    ON u.id    = d.created_by
  LEFT JOIN distributor_dispatch_items ddi ON ddi.dispatch_id = d.id
  LEFT JOIN issues i   ON i.id = ddi.issue_id
  LEFT JOIN products p ON p.id = i.product_id
  GROUP BY
    d.id, d.dispatch_no, d.dispatch_at, d.status,
    dist.name, u.name
  ORDER BY d.id DESC
  LIMIT 200
")->fetchAll();


$distributors      = fetch_distributors($pdo);
$issues_with_stock = fetch_issues_with_stock($pdo);
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<title>صرف للموزعين (DIS) — MGZ</title>
<style>
  *, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }
  body {
    font-family: 'Segoe UI', Tahoma, Arial, sans-serif;
    background: #f3f4f6;
    color: #1f2937;
    min-height: 100vh;
  }

  .back-link {
    display: inline-block;
    padding: 10px 20px;
    color: #6b7280;
    text-decoration: none;
    font-size: .9rem;
    transition: color .2s;
  }
  .back-link:hover { color: #1d4ed8; }

  .page-header {
    background: linear-gradient(135deg, #1e3a5f 0%, #2563eb 100%);
    color: #fff;
    padding: 22px 32px;
    border-radius: 12px;
    margin: 0 20px 24px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    box-shadow: 0 4px 15px rgba(37,99,235,.35);
  }
  .page-header h1 { font-size: 1.5rem; font-weight: 700; }
  .page-header .subtitle { font-size: .85rem; opacity: .8; margin-top: 4px; }

  .content { padding: 0 20px 40px; }

  .card {
    background: #fff;
    border-radius: 10px;
    box-shadow: 0 2px 8px rgba(0,0,0,.08);
    margin-bottom: 24px;
    overflow: hidden;
  }
  .card-header {
    padding: 15px 24px;
    border-bottom: 1px solid #e5e7eb;
    font-size: 1rem;
    font-weight: 600;
    color: #1e3a5f;
    display: flex;
    align-items: center;
    gap: 8px;
  }
  .card-body { padding: 24px; }

  .btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 9px 20px;
    border: none;
    border-radius: 7px;
    font-size: .9rem;
    font-weight: 600;
    cursor: pointer;
    font-family: inherit;
    transition: background .2s, transform .1s;
  }
  .btn:active { transform: scale(.97); }
  .btn-primary   { background: #2563eb; color: #fff; }
  .btn-primary:hover   { background: #1d4ed8; }

  .badge {
    display: inline-block;
    padding: 3px 12px;
    border-radius: 20px;
    font-size: .78rem;
    font-weight: 700;
  }
  .badge-draft     { background: #f3f4f6; color: #6b7280; border: 1px solid #d1d5db; }
  .badge-submitted { background: #dbeafe; color: #1e40af; }
  .badge-approved  { background: #d1fae5; color: #065f46; }
  .badge-rejected  { background: #fee2e2; color: #991b1b; }

  .table-meta { font-size: .85rem; color: #6b7280; padding: 12px 24px 0; }
  table { width: 100%; border-collapse: collapse; }
  thead th {
    background: #f1f5f9;
    color: #374151;
    font-weight: 600;
    font-size: .875rem;
    padding: 12px 18px;
    text-align: right;
    border-bottom: 2px solid #e5e7eb;
  }
  tbody tr { border-bottom: 1px solid #f3f4f6; transition: background .15s; }
  tbody tr:hover { background: #f8faff; }
  tbody tr:last-child { border-bottom: none; }
  tbody td { padding: 12px 18px; font-size: .9rem; color: #374151; text-align: right; }

  .empty-state { text-align: center; padding: 40px 20px; color: #9ca3af; }
  .empty-state .icon { font-size: 2.2rem; margin-bottom: 10px; }
  .empty-state p { font-size: .9rem; }

  .alert-danger {
    background: #fef2f2;
    border: 1px solid #fca5a5;
    border-radius: 8px;
    padding: 10px 15px;
    color: #991b1b;
    margin-bottom: 15px;
    font-size: 0.9rem;
    position: relative;
  }

  /* مودال الإنشاء */
  #dispatchModal {
    position:fixed; inset:0;
    background:rgba(0,0,0,.45);
    display:none;
    align-items:center;
    justify-content:center;
    z-index:9999;
  }
</style>
</head>
<body>

<a class="back-link" href="../dashboard.php">→ العودة للرئيسية</a>

<div class="page-header">
  <div>
    <h1>🚚 صرف للموزعين (DIS)</h1>
    <div class="subtitle">أوامر صرف المجلات للموزعين</div>
  </div>
  <div style="font-size:2rem; opacity:.2;">🚚</div>
</div>

<div class="content">

  <?php if ($error_message): ?>
    <div class="alert-danger">
      <span><?= $error_message ?></span>
      <span onclick="this.parentElement.style.display='none';"
            style="position:absolute;left:10px;top:5px;cursor:pointer;font-weight:bold;">
        ×
      </span>
    </div>
  <?php endif; ?>

  <!-- إنشاء أمر جديد (زر فقط) -->
  <div class="card">
    <div class="card-header">➕ إنشاء أمر صرف جديد</div>
    <div class="card-body">
      <button type="button" class="btn btn-primary" onclick="openDispatchModal()">
        ➕ إنشاء
      </button>
    </div>
  </div>

  <!-- قائمة الأوامر -->
  <div class="card">
    <div class="card-header">📋 آخر 200 أمر صرف</div>
    <div class="table-meta"><?= count($rows ?? []) ?> أمر</div>

    <?php if (empty($rows)): ?>
      <div class="empty-state">
        <div class="icon">🚚</div>
        <p>لا توجد أوامر صرف مسجلة بعد</p>
      </div>
    <?php else: ?>
    <table>
 <thead>
  <tr>
    <th>#</th>
    <th>رقم الأمر</th>
    <th>التاريخ</th>
    <th>الموزع</th>
    <th>الإصدارات / الكميات</th>
    <th>الحالة</th>
    <th>أنشئ بواسطة</th>
  </tr>
</thead>

      <tbody>
        <?php foreach ($rows as $idx => $r):
          [$st_ar, $st_cls] = dis_status_ar($r['status']);
        ?>
        <tr>
          <td><?= $idx + 1 ?></td>
          <td>
            <?php if ($r['dispatch_no']): ?>
              <strong><?= h($r['dispatch_no']) ?></strong>
            <?php else: ?>
              <span style="color:#9ca3af; font-style:italic;">لم يُولَّد بعد</span>
            <?php endif; ?>
          </td>
          <td><?= h($r['dispatch_at']) ?></td>
<td><?= h($r['distributor_name']) ?></td>
<td>
  <?php if (!empty($r['items_summary'])): ?>
    <?= h($r['items_summary']) ?>
  <?php else: ?>
    <span style="color:#9ca3af;font-style:italic;">لا توجد أصناف</span>
  <?php endif; ?>
</td>
<td><span class="badge <?= $st_cls ?>"><?= $st_ar ?></span></td>
<td><?= h($r['created_by_name']) ?></td>

        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
    <?php endif; ?>
  </div>

</div><!-- /content -->

<!-- مودال إنشاء أمر صرف في نفس الصفحة -->
<div id="dispatchModal">
  <div style="
    background:#fff;
    width:95%;
    max-width:500px;
    max-height:90vh;
    overflow:auto;
    border-radius:12px;
    box-shadow:0 10px 30px rgba(15,23,42,.4);
    padding:18px 18px 20px;
  ">
    <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:10px;">
      <h2 style="font-size:1rem;font-weight:700;margin:0;">إنشاء أمر صرف للموزع</h2>
      <button type="button" onclick="closeDispatchModal()"
              style="border:none;background:transparent;font-size:1.4rem;cursor:pointer;">×</button>
    </div>

    <form method="post" enctype="multipart/form-data">
      <input type="hidden" name="csrf" value="<?= h($csrf) ?>">
      <input type="hidden" name="op" value="create_and_approve">

      <div style="display:flex;flex-direction:column;gap:10px;">
        <label>
          <span style="font-size:.8rem;color:#374151;">الموزع</span>
          <select name="distributor_id" required
                  style="width:100%;padding:8px 10px;border:1px solid #d1d5db;border-radius:7px;font-size:.9rem;">
            <option value="">— اختر الموزع —</option>
            <?php foreach ($distributors as $dist): ?>
              <option value="<?= (int)$dist['id'] ?>"><?= h($dist['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </label>

        <label>
          <span style="font-size:.8rem;color:#374151;">الإصدار</span>
          <select name="issue_id" id="modal_issue_id" required
                  style="width:100%;padding:8px 10px;border:1px solid #d1d5db;border-radius:7px;font-size:.9rem;">
            <option value="">— اختر الإصدار —</option>
            <?php foreach ($issues_with_stock as $iss):
              $labelParts = [];

              $labelParts[] = $iss['product_name']; // اسم المنتج

              if (!empty($iss['issue_number'])) {
                  $labelParts[] = 'عدد ' . (int)$iss['issue_number'];
              }

              if (!empty($iss['title'])) {
                  $labelParts[] = $iss['title'];
              }

              $datePart  = str_pad((int)$iss['issue_month'], 2, '0', STR_PAD_LEFT) . '/' . $iss['issue_year'];
              $niceLabel = implode(' — ', $labelParts) . ' | ' . $datePart;
              $available = (int)$iss['available_qty'];
            ?>
              <option value="<?= (int)$iss['id'] ?>"
                      data-avail="<?= $available ?>">
                <?= h($niceLabel) ?> (المتاح: <?= $available ?>)
              </option>
            <?php endforeach; ?>
          </select>
        </label>

        <div id="modal_available_box" style="
          font-size:.8rem;
          color:#0f766e;
          background:#ecfdf5;
          border:1px solid #a7f3d0;
          border-radius:8px;
          padding:8px 10px;
          display:none;
        ">
          المتاح من هذا الإصدار بالمخزن: <span id="modal_available_qty" style="font-weight:700;"></span>
        </div>

        <label>
          <span style="font-size:.8rem;color:#374151;">الكمية</span>
          <input type="number" name="qty" min="1" step="1" required
                 style="width:100%;padding:8px 10px;border:1px solid #d1d5db;border-radius:7px;font-size:.9rem;">
        </label>

        <label>
          <span style="font-size:.8rem;color:#374151;">صورة الإيصال (POD)</span>
          <input type="file" name="pod" accept=".jpg,.jpeg,.png,.pdf"
                 style="width:100%;font-size:.85rem;">
        </label>
      </div>

      <div style="display:flex;justify-content:space-between;gap:10px;flex-wrap:wrap;margin-top:14px;">
        <button type="submit"
                style="flex:1 1 160px;background:#16a34a;color:#fff;border:none;border-radius:8px;padding:8px 12px;font-size:.9rem;font-weight:600;cursor:pointer;">
          ✔ اعتماد وإرسال
        </button>
        <button type="button" onclick="closeDispatchModal()"
                style="flex:1 1 120px;background:#f3f4f6;color:#374151;border:none;border-radius:8px;padding:8px 12px;font-size:.9rem;font-weight:600;cursor:pointer;">
          إلغاء
        </button>
      </div>
    </form>
  </div>
</div>

<script>
function openDispatchModal() {
  document.getElementById('dispatchModal').style.display = 'flex';
}
function closeDispatchModal() {
  document.getElementById('dispatchModal').style.display = 'none';
}

/* عند اختيار الإصدار، إظهار الرصيد المتاح فورًا من data-avail */
const issueSelect = document.getElementById('modal_issue_id');
const availBox    = document.getElementById('modal_available_box');
const availSpan   = document.getElementById('modal_available_qty');

if (issueSelect) {
  issueSelect.addEventListener('change', function() {
    const opt   = this.options[this.selectedIndex];
    const avail = opt ? opt.getAttribute('data-avail') : null;
    if (!avail || isNaN(avail)) {
      availBox.style.display = 'none';
      availSpan.textContent  = '';
    } else {
      availSpan.textContent  = avail;
      availBox.style.display = 'block';
    }
  });
}
</script>

</body>
</html>
