<?php
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/docno.php';
require_once __DIR__ . '/../app/upload.php';
require_once __DIR__ . '/../app/ledger.php';
require_once __DIR__ . '/../app/audit.php';

$me  = require_role(['ADMIN','DISTRIBUTOR']);
$pdo = db();
$csrf = csrf_token();
$action = $_GET['action'] ?? 'list';

function json_out($data, int $code=200): void {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}

function my_distributor_id(array $me): ?int {
  if ($me['role'] === 'DISTRIBUTOR') {
    return !empty($me['distributor_id']) ? (int)$me['distributor_id'] : null;
  }
  return null;
}

function fetch_delivery(PDO $pdo, int $id, ?int $restrictDistributorId=null): ?array {
  $sql = "SELECT od.*, o.name AS outlet_name, d.name AS distributor_name
          FROM outlet_deliveries od
          JOIN outlets o ON o.id=od.outlet_id
          JOIN distributors d ON d.id=od.distributor_id
          WHERE od.id=? AND od.status='APPROVED' AND od.deal_type='CONSIGNMENT'";
  $args = [$id];
  if ($restrictDistributorId !== null) { $sql .= " AND od.distributor_id=?"; $args[] = $restrictDistributorId; }
  $st = $pdo->prepare($sql);
  $st->execute($args);
  $r = $st->fetch();
  return $r ?: null;
}

function fetch_delivery_items(PDO $pdo, int $deliveryId): array {
  $st = $pdo->prepare("SELECT odi.*, p.name AS product_name, i.issue_number, i.issue_month, i.issue_year, i.cover_price
                       FROM outlet_delivery_items odi
                       JOIN issues i ON i.id=odi.issue_id
                       JOIN products p ON p.id=i.product_id
                       WHERE odi.delivery_id=?");
  $st->execute([$deliveryId]);
  return $st->fetchAll();
}

function settlement_exists(PDO $pdo, int $deliveryId): bool {
  // يعتمد على وجود العمود delivery_id في outlet_settlements
  try {
    $st = $pdo->prepare("SELECT COUNT(*) FROM outlet_settlements WHERE delivery_id=?");
    $st->execute([$deliveryId]);
    return (int)$st->fetchColumn() > 0;
  } catch (Throwable $e) {
    return false;
  }
}

function distributor_commission(PDO $pdo, int $distributorId, float $grossAmount): float {
  $st = $pdo->prepare("SELECT commission_type, commission_value FROM distributors WHERE id=?");
  $st->execute([$distributorId]);
  $d = $st->fetch();
  if (!$d) return 0.0;
  if ($d['commission_type'] === 'PERCENT') return round($grossAmount * max(0.0, (float)$d['commission_value']) / 100.0, 3);
  if ($d['commission_type'] === 'FIXED')   return round(max(0.0, (float)$d['commission_value']), 3);
  return 0.0;
}

$distId = my_distributor_id($me);
// ensure patch applied (delivery_id column)
$hasDeliveryCol = true;
try { $pdo->query("SELECT delivery_id FROM outlet_settlements LIMIT 1"); } catch (Throwable $e) { $hasDeliveryCol = false; }


// API: load delivery for modal
if ($action === 'api_delivery') {
  if (!$hasDeliveryCol) json_out(["ok"=>false,"error"=>"يجب تشغيل DB_PATCH_SETL.sql أولاً (إضافة delivery_id)"], 500);

  $id = (int)($_GET['id'] ?? 0);
  if ($id<=0) json_out(['ok'=>false,'error'=>'Bad id'], 400);
  $d = fetch_delivery($pdo, $id, $distId);
  if (!$d) json_out(['ok'=>false,'error'=>'Not found'], 404);
  if (settlement_exists($pdo, $id)) json_out(['ok'=>false,'error'=>'تمت تسوية هذه المعاملة بالفعل'], 409);
  $items = fetch_delivery_items($pdo, $id);
  json_out(['ok'=>true,'delivery'=>$d,'items'=>$items]);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $op = $_POST['op'] ?? '';

  if ($op === 'settle_delivery') {
    if ($distId === null) exit('Only distributor');
    $delivery_id = (int)($_POST['delivery_id'] ?? 0);
    $cash_collected = (float)($_POST['cash_collected'] ?? 0);
    $cash_collected = max(0.0, $cash_collected);

    $d = fetch_delivery($pdo, $delivery_id, $distId);
    if (!$d) exit('Delivery not found');
    if (settlement_exists($pdo, $delivery_id)) exit('Already settled');

    $issue_ids = $_POST['issue_id'] ?? [];
    $sold      = $_POST['qty_sold'] ?? [];
    $ret       = $_POST['qty_returned'] ?? [];
    $dam       = $_POST['qty_damaged'] ?? [];

    if (!is_array($issue_ids) || count($issue_ids) === 0) exit('No lines');

    // load delivered quantities
    $delivered = [];
    foreach (fetch_delivery_items($pdo, $delivery_id) as $it) {
      $delivered[(int)$it['issue_id']] = (int)$it['qty_delivered'];
    }

    $lines = [];
    $hasDamage = false;
    for ($k=0; $k<count($issue_ids); $k++) {
      $iid = (int)$issue_ids[$k];
      if ($iid<=0) continue;
      $qs = (int)($sold[$k] ?? 0);
      $qr = (int)($ret[$k] ?? 0);
      $qd = (int)($dam[$k] ?? 0);
      if ($qs<0 || $qr<0 || $qd<0) exit('Bad qty');
      $orig = $delivered[$iid] ?? 0;
      if ($orig<=0) exit('Issue not in delivery');
      if (($qs+$qr+$qd) !== $orig) exit('يجب أن يكون (تحصيل + مرتجع + تالف) = الكمية الأصلية لكل سطر');
      if ($qd>0) $hasDamage = true;
      $lines[] = ['issue_id'=>$iid,'qty_sold'=>$qs,'qty_returned'=>$qr,'qty_damaged'=>$qd,'orig'=>$orig];
    }
    if (!$lines) exit('No valid lines');

    // damage photo required if any damage
    $damagePhotoPath = null;
    if ($hasDamage) {
      if (empty($_FILES['damage_photo']) || !is_array($_FILES['damage_photo']) || ($_FILES['damage_photo']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
        exit('صورة التالف إجبارية عند وجود تالف');
      }
      $attId = save_upload($_FILES['damage_photo'], (int)$me['id']);
      $st = $pdo->prepare("SELECT path FROM attachments WHERE id=?");
      $st->execute([$attId]);
      $damagePhotoPath = (string)$st->fetchColumn();
    }

    $pdo->beginTransaction();
    try {
      $settleAt = date('Y-m-d H:i:s');
      $settleNo = next_doc_no($pdo, 'OST', date('Y-m-d'));
      $status = $hasDamage ? 'SUBMITTED' : 'APPROVED';

      // requires column delivery_id
      $pdo->prepare("INSERT INTO outlet_settlements (settlement_no, distributor_id, outlet_id, delivery_id, settlement_at, created_by, status)
                     VALUES (?,?,?,?,?,?,?)")
          ->execute([$settleNo, (int)$d['distributor_id'], (int)$d['outlet_id'], $delivery_id, $settleAt, (int)$me['id'], $status]);
      $settlementId = (int)$pdo->lastInsertId();

      // create invoice for sold qty (always), even if damage pending
      $invNo = next_doc_no($pdo, 'INV', date('Y-m-d'));
      $pdo->prepare("INSERT INTO invoices (invoice_no, outlet_id, distributor_id, source_type, source_id, issued_at, status, currency, total_amount, total_paid, balance_due, created_by)
                     VALUES (?,?,?,'CONSIGNMENT_SETTLEMENT',?,NOW(),'ISSUED','OMR',0,0,0,?)")
          ->execute([$invNo, (int)$d['outlet_id'], (int)$d['distributor_id'], $settlementId, (int)$me['id']]);
      $invoiceId = (int)$pdo->lastInsertId();

      $total = 0.0;
      foreach ($lines as $ln) {
        $iid = (int)$ln['issue_id'];
        $qs = (int)$ln['qty_sold'];
        $qr = (int)$ln['qty_returned'];
        $qd = (int)$ln['qty_damaged'];

        // ledger movements:
        // sold/returned/damaged all leave OUTLET
        if ($qs>0) ledger_post($pdo, 'OUTLET', (int)$d['outlet_id'], $iid, 0, $qs, 'SETTLEMENT', $settlementId, (int)$me['id'], 'Sold to customers');
        if ($qr>0) {
          ledger_post($pdo, 'OUTLET', (int)$d['outlet_id'], $iid, 0, $qr, 'SETTLEMENT', $settlementId, (int)$me['id'], 'Returned to distributor');
          ledger_post($pdo, 'DISTRIBUTOR', (int)$d['distributor_id'], $iid, $qr, 0, 'SETTLEMENT', $settlementId, (int)$me['id'], 'Return from outlet');
        }
        if ($qd>0) {
          ledger_post($pdo, 'OUTLET', (int)$d['outlet_id'], $iid, 0, $qd, 'DAMAGE_PENDING', $settlementId, (int)$me['id'], 'Damaged pending approval');
          // ضع الكمية التالفة على عهدة الموزع (قيد الاعتماد)
          ledger_post($pdo, 'DISTRIBUTOR', (int)$d['distributor_id'], $iid, $qd, 0, 'DAMAGE_PENDING', $settlementId, (int)$me['id'], 'Damaged held with distributor pending approval');
          // سجل طلب تالف (قيد المراجعة)
          $pdo->prepare("INSERT INTO damage_reports (issue_id, distributor_id, outlet_id, location, qty, photo_path, status, created_by, created_at)
                         VALUES (?,?,?, 'WITH_OUTLET', ?, ?, 'PENDING', ?, NOW())")
              ->execute([$iid, (int)$d['distributor_id'], (int)$d['outlet_id'], $qd, $damagePhotoPath, (int)$me['id']]);
        }

        // line pricing
        $st = $pdo->prepare("SELECT cover_price FROM issues WHERE id=?");
        $st->execute([$iid]);
        $unit = (float)$st->fetchColumn();
        $gross = round($qs * $unit, 3);
        $comm  = distributor_commission($pdo, (int)$d['distributor_id'], $gross);
        $net   = round(max(0.0, $gross - $comm), 3);

        $pdo->prepare("INSERT INTO outlet_settlement_lines (settlement_id, issue_id, qty_sold, qty_returned, qty_damaged, unit_price_applied, gross_amount, distributor_commission, net_due)
                       VALUES (?,?,?,?,?,?,?,?,?)")
            ->execute([$settlementId, $iid, $qs, $qr, $qd, $unit, $gross, $comm, $net]);

        if ($qs>0) {
          $lineTotal = $gross;
          $total += $lineTotal;
          $pdo->prepare("INSERT INTO invoice_items (invoice_id, issue_id, qty, unit_price, line_total, commission_amount) VALUES (?,?,?,?,?,?)")
              ->execute([$invoiceId, $iid, $qs, $unit, $lineTotal, $comm]);
        }
      }

      $total = round($total, 3);
      $paid = 0.0;
      $balance = $total;

      if ($cash_collected > 0.0) {
        $payNo = next_doc_no($pdo, 'PAY', date('Y-m-d'));
        $pdo->prepare("INSERT INTO payments (payment_no, outlet_id, distributor_id, amount, method, paid_at, notes, created_by)
                       VALUES (?,?,?,?,'CASH',NOW(),?,?)")
            ->execute([$payNo, (int)$d['outlet_id'], (int)$d['distributor_id'], $cash_collected, 'تحصيل تسوية أمانة', (int)$me['id']]);
        $payId = (int)$pdo->lastInsertId();
        $alloc = min($cash_collected, $total);
        if ($alloc > 0) {
          $pdo->prepare("INSERT INTO payment_allocations (payment_id, invoice_id, amount_allocated) VALUES (?,?,?)")
              ->execute([$payId, $invoiceId, $alloc]);
          $paid = round($alloc, 3);
          $balance = round(max(0.0, $total - $paid), 3);
        }
      }

      $pdo->prepare("UPDATE invoices SET total_amount=?, total_paid=?, balance_due=? WHERE id=?")
          ->execute([$total, $paid, $balance, $invoiceId]);

      audit_log($pdo, (int)$me['id'], 'CREATE', 'outlet_settlements', $settlementId, ['delivery_id'=>$delivery_id,'has_damage'=>$hasDamage]);

      $pdo->commit();
      flash($hasDamage ? 'تم تنفيذ التسوية (التالف بانتظار موافقة الإدارة)' : 'تم تنفيذ التسوية بنجاح');
      redirect('settlements.php');

    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }
  if (!$hasDeliveryCol) {
	    echo '<div style="font-family:Tahoma; padding:16px;">'	       .'<h3>مطلوب تحديث قاعدة البيانات</h3>'	       .'<p>شغّل الملف <b>DB_PATCH_SETL.sql</b> مرة واحدة داخل phpMyAdmin ثم أعد فتح الصفحة.</p>'	       .'</div>';
    exit;
  }

}

// Page list
$rows = [];
if ($me['role'] === 'DISTRIBUTOR') {
  if (!$distId) exit('Distributor user not linked');
  // معاملات أمانة تحتاج تسوية: CONSIGNMENT + APPROVED + لا يوجد settlement
  $st = $pdo->prepare("SELECT od.id, od.delivery_no, od.delivery_at, o.name AS outlet_name
                       FROM outlet_deliveries od
                       JOIN outlets o ON o.id=od.outlet_id
                       LEFT JOIN outlet_settlements os ON os.delivery_id = od.id
                       WHERE od.distributor_id=? AND od.deal_type='CONSIGNMENT' AND od.status='APPROVED' AND os.id IS NULL
                       ORDER BY od.id DESC
                       LIMIT 200");
  $st->execute([$distId]);
  $rows = $st->fetchAll();
} else {
  // Admin: قائمة طلبات التالف المعلقة
}

?><!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>تسويات الأمانة — MGZ</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<style>
  body{font-family:Tahoma,Arial; background:#f4f7f6; margin:0; padding:16px;}
  .topbar{display:flex; align-items:center; justify-content:space-between; gap:10px; flex-wrap:wrap; margin-bottom:12px;}
  .btn{background:#1a73e8; color:#fff; border:none; padding:10px 14px; border-radius:10px; cursor:pointer; font-size:14px;}
  .btn-secondary{background:#6c757d;}
  .card{background:#fff; border-radius:14px; padding:14px; box-shadow:0 2px 10px rgba(0,0,0,.06);}
  table{width:100%; border-collapse:collapse;}
  th,td{padding:10px; border-bottom:1px solid #eee; font-size:13px; text-align:right;}
  th{background:#fafafa;}
  .muted{color:#666; font-size:12px;}
  .linkbtn{border:none; background:#f1f5ff; padding:8px 10px; border-radius:10px; cursor:pointer;}
  .modal-backdrop{position:fixed; inset:0; background:rgba(0,0,0,.55); display:none; align-items:center; justify-content:center; z-index:9999;}
  .modal{background:#fff; width:min(980px, 96vw); max-height:90vh; overflow:auto; border-radius:16px; padding:14px;}
  @media (max-width:520px){ .modal{width:98vw; height:92vh; max-height:92vh;} .btn{width:100%;} }
  .modal-header{display:flex; justify-content:space-between; align-items:center; gap:10px; margin-bottom:10px;}
  .x{background:transparent; border:none; font-size:22px; cursor:pointer;}
  input{width:100%; padding:10px; border:1px solid #ddd; border-radius:10px; font-size:14px;}
</style>
</head>
<body>

<div class="topbar">
  <a href="../dashboard.php" class="btn btn-secondary" style="text-decoration:none;">⬅ العودة</a>
  <div style="flex:1"></div>
</div>

<?php show_flash(); ?>

<div class="card">
  <h3 style="margin:0 0 10px 0;">🧾 معاملات أمانة تحتاج تسوية</h3>
  <table>
    <thead>
      <tr><th>#</th><th>المنفذ</th><th>التاريخ</th><th></th></tr>
    </thead>
    <tbody>
      <?php if(!$rows): ?>
        <tr><td colspan="4" class="muted">لا يوجد معاملات تحتاج تسوية</td></tr>
      <?php else: foreach($rows as $r): ?>
        <tr>
          <td><?=h($r['delivery_no'] ?: $r['id'])?></td>
          <td><?=h($r['outlet_name'])?></td>
          <td><?=h(date('Y-m-d H:i', strtotime($r['delivery_at'])))?></td>
          <td><button class="linkbtn" data-settle="<?= (int)$r['id'] ?>">تسوية</button></td>
        </tr>
      <?php endforeach; endif; ?>
    </tbody>
  </table>
</div>

<!-- Settlement Modal -->
<div class="modal-backdrop" id="mSettle">
  <div class="modal" role="dialog" aria-modal="true">
    <div class="modal-header">
      <div>
        <div style="font-weight:700;">تنفيذ تسوية أمانة</div>
        <div class="muted" id="mInfo">...</div>
      </div>
      <button class="x" data-close="#mSettle">×</button>
    </div>

    <form method="post" enctype="multipart/form-data" id="frmSettle">
      <input type="hidden" name="csrf" value="<?=h($csrf)?>">
      <input type="hidden" name="op" value="settle_delivery">
      <input type="hidden" name="delivery_id" id="deliveryId">

      <div class="card" style="margin-bottom:10px;">
        <div class="muted">لكل سطر: (تحصيل + مرتجع + تالف) لازم يساوي الكمية الأصلية</div>
      </div>

      <div class="card" style="padding:0; overflow:hidden;">
        <table id="tblLines">
          <thead>
            <tr>
              <th>المنتج</th><th>الإصدار</th><th>الأصلية</th>
              <th>تحصيل</th><th>مرتجع</th><th>تالف</th>
            </tr>
          </thead>
          <tbody id="linesBody"></tbody>
        </table>
      </div>

      <div style="margin-top:10px;" class="card">
        <label class="muted">تحصيل نقدي (اختياري)</label>
        <input type="number" step="0.001" min="0" name="cash_collected" value="0">
        <div class="muted" style="margin-top:6px;">لو أدخلت رقم هنا، هيتسجل Payment ويتخصص للفاتورة الناتجة من التسوية.</div>
      </div>

      <div style="margin-top:10px;" class="card" id="damageBox" style="display:none;">
        <label class="muted">صورة التالف (إجباري لو في تالف)</label>
        <input type="file" name="damage_photo" id="damagePhoto" accept="image/*">
      </div>

      <div style="margin-top:12px; display:flex; gap:10px; flex-wrap:wrap;">
        <button type="submit" class="btn" id="btnDo" disabled>تنفيذ التسوية</button>
        <button type="button" class="btn btn-secondary" data-close="#mSettle">إلغاء</button>
      </div>

    </form>
  </div>
</div>

<script>
const $ = (s, el=document) => el.querySelector(s);
const $$ = (s, el=document) => Array.from(el.querySelectorAll(s));

function openModal(id){ document.getElementById(id).style.display='flex'; }
function closeModal(sel){ const m=document.querySelector(sel); if(m) m.style.display='none'; }

document.addEventListener('click', (e)=>{ const c=e.target.getAttribute('data-close'); if(c) closeModal(c); });

$$('[data-settle]').forEach(btn=>{
  btn.addEventListener('click', async ()=>{
    const id = btn.getAttribute('data-settle');
    openModal('mSettle');
    $('#linesBody').innerHTML = '<tr><td colspan="6" class="muted">...</td></tr>';
    const r = await fetch('settlements.php?action=api_delivery&id='+encodeURIComponent(id));
    const js = await r.json();
    if(!js.ok){ $('#linesBody').innerHTML = `<tr><td colspan="6" class="muted">${escapeHtml(js.error||'خطأ')}</td></tr>`; return; }
    $('#deliveryId').value = id;
    $('#mInfo').textContent = `المنفذ: ${js.delivery.outlet_name} — التاريخ: ${js.delivery.delivery_at}`;
    renderLines(js.items);
    validate();
  });
});

function renderLines(items){
  const tb = $('#linesBody');
  tb.innerHTML='';
  items.forEach(it=>{
    const label = `عدد ${it.issue_number||''} - ${String(it.issue_month).padStart(2,'0')}/${it.issue_year}`;
    const tr = document.createElement('tr');
    tr.dataset.orig = String(it.qty_delivered);
    tr.innerHTML = `
      <td>${escapeHtml(it.product_name)}</td>
      <td>${escapeHtml(label)}</td>
      <td>${it.qty_delivered}</td>
      <td><input type="number" min="0" step="1" value="0" name="qty_sold[]" class="qSold"></td>
      <td><input type="number" min="0" step="1" value="0" name="qty_returned[]" class="qRet"></td>
      <td><input type="number" min="0" step="1" value="0" name="qty_damaged[]" class="qDam"></td>
      <input type="hidden" name="issue_id[]" value="${it.issue_id}">
    `;
    tb.appendChild(tr);
    ['qSold','qRet','qDam'].forEach(cls=>{
      tr.querySelector('.'+cls).addEventListener('input', ()=>{ clampRow(tr); validate(); });
    });
  });
}

function clampRow(tr){
  const orig = Number(tr.dataset.orig||0);
  const s = tr.querySelector('.qSold');
  const r = tr.querySelector('.qRet');
  const d = tr.querySelector('.qDam');
  let vs = Math.max(0, Math.floor(Number(s.value||0)));
  let vr = Math.max(0, Math.floor(Number(r.value||0)));
  let vd = Math.max(0, Math.floor(Number(d.value||0)));
  // clamp each to orig
  if (vs>orig) vs=orig;
  if (vr>orig) vr=orig;
  if (vd>orig) vd=orig;
  s.value = String(vs); r.value = String(vr); d.value = String(vd);
}

function validate(){
  let ok = true;
  let hasDamage = false;
  $$('#linesBody tr').forEach(tr=>{
    const orig = Number(tr.dataset.orig||0);
    const vs = Number(tr.querySelector('.qSold').value||0);
    const vr = Number(tr.querySelector('.qRet').value||0);
    const vd = Number(tr.querySelector('.qDam').value||0);
    if ((vs+vr+vd) !== orig) ok = false;
    if (vd>0) hasDamage = true;
  });

  const box = document.getElementById('damageBox');
  if (hasDamage) {
    box.style.display='block';
    const f = document.getElementById('damagePhoto');
    if (!f.files || f.files.length===0) ok = false;
  } else {
    box.style.display='none';
  }

  document.getElementById('btnDo').disabled = !ok;
}

$('#damagePhoto')?.addEventListener('change', ()=>validate());

document.getElementById('frmSettle')?.addEventListener('submit', (e)=>{
  // validation already handled
});

function escapeHtml(s){
  return String(s??'').replace(/[&<>"']/g, c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c]));
}
</script>

</body>
</html>
