<?php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/security.php';

function auth_start(): void {
  if (session_status() === PHP_SESSION_ACTIVE) return;

  // Apply baseline security headers on every request.
  mgz_apply_security_headers();

  $forceHttps = (getenv('FORCE_HTTPS') === '1');
  $secure = $forceHttps || (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');

  $sameSite = (getenv('SESSION_SAMESITE') ?: 'Strict');
  $lifetime = 0;
  $path = '/';
  $domain = '';

  // ✅ PHP 7.3+ supports array options
  if (defined('PHP_VERSION_ID') && PHP_VERSION_ID >= 70300) {
    session_set_cookie_params([
      'lifetime' => $lifetime,
      'path' => $path,
      'domain' => $domain,
      'secure' => $secure,
      'httponly' => true,
      'samesite' => $sameSite,
    ]);
  } else {
    // ✅ PHP < 7.3 fallback (SameSite via path trick)
    // Works on most browsers/servers for legacy PHP.
    session_set_cookie_params(
      $lifetime,
      $path . '; samesite=' . $sameSite,
      $domain,
      $secure,
      true
    );
  }

  session_start();
}

function auth_user(): ?array {
  auth_start();
  $timeout = (int)(getenv('SESSION_IDLE_TIMEOUT') ?: 3600);
  $now = time();
  if (!empty($_SESSION['last_activity']) && ($now - (int)$_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    return null;
  }
  $_SESSION['last_activity'] = $now;

  auth_start();
  return $_SESSION['user'] ?? null;
}

function require_login(): array {
  $u = auth_user();
  if (!$u) {
    header("Location: login.php");
    exit;
  }
  return $u;
}

function require_role(array $roles): array {
  $u = require_login();
  if (!in_array($u['role'], $roles, true)) {
    http_response_code(403);
    exit("Forbidden");
  }
  return $u;
}