<?php
require_once __DIR__ . '/env.php';
require_once __DIR__ . '/logger.php';
mgz_load_env_once();

$APP_ENV = getenv('APP_ENV') ?: 'production';
$IS_DEV = ($APP_ENV === 'development' || $APP_ENV === 'dev');

// Error handling: never show errors to end users in production
ini_set('display_errors', $IS_DEV ? '1' : '0');
ini_set('display_startup_errors', $IS_DEV ? '1' : '0');
error_reporting(E_ALL);

// Convert warnings/notices to exceptions to avoid silent failures
set_error_handler(function($severity, $message, $file, $line) {
  if (!(error_reporting() & $severity)) return false;
  throw new ErrorException($message, 0, $severity, $file, $line);
});

// Exception handler: log details, show generic message in production
set_exception_handler(function($e) use ($IS_DEV) {
  mgz_log('ERROR', $e->getMessage(), [
    'type' => get_class($e),
    'file' => $e->getFile(),
    'line' => $e->getLine(),
    'trace' => $e->getTraceAsString(),
  ]);

  http_response_code(500);
  if ($IS_DEV) {
    echo "<pre style='direction:ltr;text-align:left;white-space:pre-wrap;background:#111;color:#fff;padding:12px;border-radius:10px;'>";
    echo htmlspecialchars((string)$e);
    echo "</pre>";
  } else {
    echo "حدث خطأ غير متوقع. برجاء المحاولة لاحقاً.";
  }
  exit;
});

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;

  $cfg = require __DIR__ . '/config.php';
  $db = $cfg['db'];

  $dsn = "mysql:host={$db['host']};dbname={$db['name']};charset={$db['charset']}";
  $pdo = new PDO($dsn, $db['user'], $db['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::ATTR_EMULATE_PREPARES => false,
  ]);
  return $pdo;
}
