<?php
if (!function_exists('h')) {
    function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

if (!function_exists('fmt_int')) {
    /** Format integers without thousands separators (e.g., 2000 not 2,000) */
    function fmt_int($n): string {
        return number_format((float)$n, 0, '.', '');
    }
}
if (!function_exists('fmt_num')) {
    /** Format numbers with up to 3 decimals, no thousands separators, trimming trailing zeros */
    function fmt_num($n, int $decimals = 3): string {
        $s = number_format((float)$n, $decimals, '.', '');
        $s = rtrim(rtrim($s, '0'), '.');
        return $s === '' ? '0' : $s;
    }
}

if (!function_exists('omr')) {
    function omr($n) {
        // OMR formatting (Oman): no thousands separators, no excessive zeros
        $s = fmt_num($n, 3);
        return $s . ' ر.ع';
    }
}
if (!function_exists('date_ar')) {
    function date_ar($d) { return $d ? date('Y/m/d', strtotime($d)) : '—'; }
}
if (!function_exists('status_label')) {
    function status_label($s) {
        $map = [
            'DRAFT'    => ['مسودة','secondary'],
            'SUBMITTED'=> ['مرفوعة','warning'],
            'APPROVED' => ['معتمدة','success'],
            'REJECTED' => ['مرفوضة','danger'],
            'ISSUED'   => ['صادرة','primary'],
            'PARTIAL'  => ['مدفوعة جزئياً','warning'],
            'PAID'     => ['مدفوعة','success'],
            'CANCELLED'=> ['ملغاة','danger'],
            'OPEN'     => ['مفتوح','success'],
            'CLOSED'   => ['مغلق','secondary'],
            'PENDING'  => ['قيد المراجعة','warning'],
            'FULFILLED'=> ['تم التنفيذ','success'],
        ];
        $v = $map[$s] ?? [$s,'secondary'];
        return '<span class="badge bg-'.$v[1].'">'.$v[0].'</span>';
    }
}
if (!function_exists('deal_label')) {
    function deal_label($d) {
        $map = ['CONSIGNMENT'=>'أمانة','DIRECT_SALE'=>'بيع مباشر','FREE'=>'مجاني'];
        return $map[$d] ?? $d;
    }
}
if (!function_exists('flash')) {
    function flash($msg, $type='success') {
        $_SESSION['flash'] = ['msg'=>$msg,'type'=>$type];
    }
}
if (!function_exists('get_flash')) {
    function get_flash() {
        if (!empty($_SESSION['flash'])) {
            $f = $_SESSION['flash'];
            unset($_SESSION['flash']);
            return $f;
        }
        return null;
    }
}
if (!function_exists('show_flash')) {
    function show_flash() {
        $f = get_flash();
        if ($f) echo '<div class="alert alert-'.$f['type'].'" role="alert">'.h($f['msg']).'</div>';
    }
}
if (!function_exists('redirect')) {
    function redirect(string $url) {
        if (!headers_sent()) {
            header("Location: $url");
            exit;
        } else {
            echo '<script>window.location.href='.json_encode($url).';</script>';
            exit;
        }
    }
}

/*
 * ─────────────────────────────────────────────────────────────
 * Invoice totals helpers
 *
 * Business rule for this project:
 * - When the distributor collects cash from the outlet, the outlet has effectively paid.
 * - The payment may remain HELD until the distributor remits it to the office.
 * Therefore, for invoice "paid" calculations we count allocations from payments in
 * status HELD and POSTED.
 *
 * NOTE: "POSTED" is still used to indicate the cash has been remitted to the office.
 */

if (!function_exists('invoice_allocated_sum')) {
    /** Sum of allocations on an invoice for payments in HELD/POSTED (or only POSTED when $includeHeld=false). */
    function invoice_allocated_sum(PDO $pdo, int $invoiceId, bool $includeHeld = true): float {
        $statuses = $includeHeld ? "('HELD','POSTED')" : "('POSTED')";
        $st = $pdo->prepare(
            "SELECT COALESCE(SUM(a.amount_allocated),0) AS paid\n"
          . "FROM payment_allocations a\n"
          . "JOIN payments p ON p.id=a.payment_id\n"
          . "WHERE a.invoice_id=? AND p.status IN $statuses"
        );
        $st->execute([$invoiceId]);
        $row = $st->fetch();
        return round((float)($row['paid'] ?? 0), 3);
    }
}


if (!function_exists('invoice_paid_by_outlet')) {
    /** Amount considered paid by outlet (HELD + POSTED). */
    function invoice_paid_by_outlet(PDO $pdo, int $invoiceId): float {
        return invoice_allocated_sum($pdo, $invoiceId, true);
    }
}

if (!function_exists('invoice_remitted_to_office')) {
    /** Amount actually remitted to office (POSTED only). */
    function invoice_remitted_to_office(PDO $pdo, int $invoiceId): float {
        return invoice_allocated_sum($pdo, $invoiceId, false);
    }
}

if (!function_exists('recompute_invoice_totals')) {
    /** Recompute invoice totals (total_paid, balance_due, status) based on payment allocations. */
    function recompute_invoice_totals(PDO $pdo, int $invoiceId, bool $includeHeld = true): void {
        $st = $pdo->prepare("SELECT total_amount FROM invoices WHERE id=?");
        $st->execute([$invoiceId]);
        $row = $st->fetch();
        if (!$row) return;

        $total = round((float)$row['total_amount'], 3);
        $paid  = invoice_allocated_sum($pdo, $invoiceId, $includeHeld);
        $bal   = round(max(0.0, $total - $paid), 3);
        $status = ($bal <= 0.0) ? 'PAID' : (($paid > 0.0) ? 'PARTIAL' : 'ISSUED');

        $pdo->prepare("UPDATE invoices SET total_paid=?, balance_due=?, status=? WHERE id=?")
            ->execute([$paid, $bal, $status, $invoiceId]);
    }
}
