<?php
require_once __DIR__ . '/db.php';

if (!function_exists('ledger_post')) {
    /**
     * Post a stock movement to inventory_ledger.
     */
    function ledger_post(PDO $pdo, string $entityType, $entityId, int $issueId, int $qtyIn, int $qtyOut,
                         string $refType, int $refId, ?int $createdBy = null, ?string $note = null,
                         string $bucket = 'AVAILABLE', string $refLine = 'MAIN'): int {
        if ($entityType === 'WAREHOUSE') {
            $entityId = null; // enforce policy
        }
        $st = $pdo->prepare("INSERT INTO inventory_ledger
            (entity_type, entity_id, bucket, issue_id, qty_in, qty_out, ref_type, ref_id, ref_line, status, note, created_by, created_at)
            VALUES (?,?,?,?,?,?,?,?,?,'POSTED',?,?,NOW())
            ON DUPLICATE KEY UPDATE
              id = LAST_INSERT_ID(id),
              note = VALUES(note)");
        try {
            $st->execute([
                $entityType,
                $entityId,
                $bucket,
                $issueId,
                max(0, $qtyIn),
                max(0, $qtyOut),
                $refType,
                $refId,
                $refLine,
                $note,
                $createdBy,
            ]);
        } catch (PDOException $e) {
            // If a unique constraint other than idempotency fires, bubble it up
            throw $e;
        }
        return (int)$pdo->lastInsertId();
    }
}

if (!function_exists('ledger_balance')) {
    /**
     * Current balance for entity/issue using v_inventory_balance.
     */
    function ledger_balance(PDO $pdo, string $entityType, $entityId, int $issueId): int {
        if ($entityType === 'WAREHOUSE') $entityId = null;
        $st = $pdo->prepare("SELECT COALESCE(balance_qty,0) FROM v_inventory_balance WHERE entity_type=? AND ((? IS NULL AND entity_id IS NULL) OR entity_id=?) AND issue_id=?");
        $st->execute([$entityType, $entityId, $entityId, $issueId]);
        return (int)$st->fetchColumn();
    }
}

if (!function_exists('ledger_balance_bucket')) {
    /**
     * Current balance for a specific bucket directly from inventory_ledger.
     * Useful for buckets that are not included in v_inventory_balance (eg. HOLD).
     */
    function ledger_balance_bucket(PDO $pdo, string $entityType, $entityId, int $issueId, string $bucket): int {
        if ($entityType === 'WAREHOUSE') $entityId = null;
        $st = $pdo->prepare("SELECT COALESCE(SUM(CASE WHEN status='POSTED' AND bucket=? THEN (qty_in-qty_out) ELSE 0 END),0)
                             FROM inventory_ledger
                             WHERE entity_type=? AND ((? IS NULL AND entity_id IS NULL) OR entity_id=?) AND issue_id=?");
        $st->execute([$bucket, $entityType, $entityId, $entityId, $issueId]);
        return (int)$st->fetchColumn();
    }
}

