<?php
require_once __DIR__ . '/db.php';

/**
 * Simple brute-force protection.
 *
 * Defaults:
 * - window: 15 minutes
 * - max attempts: 5
 * - lockout: 15 minutes
 */

function mgz_client_ip(): string {
  // If behind a reverse proxy, set TRUST_PROXY=1 and provide X-Forwarded-For.
  if (getenv('TRUST_PROXY') === '1' && !empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
    $parts = explode(',', (string)$_SERVER['HTTP_X_FORWARDED_FOR']);
    return trim((string)$parts[0]);
  }
  return (string)($_SERVER['REMOTE_ADDR'] ?? '');
}

function mgz_login_is_locked(PDO $pdo, string $username, string $ip): array {
  $max = (int)(getenv('LOGIN_MAX_ATTEMPTS') ?: 5);
  $windowMin = (int)(getenv('LOGIN_WINDOW_MIN') ?: 15);
  $lockMin = (int)(getenv('LOGIN_LOCK_MIN') ?: 15);

  $st = $pdo->prepare("SELECT locked_until FROM login_attempts WHERE username=? AND ip=? LIMIT 1");
  $st->execute([$username, $ip]);
  $lockedUntil = $st->fetchColumn();
  if ($lockedUntil) {
    $ts = strtotime((string)$lockedUntil);
    if ($ts !== false && $ts > time()) {
      return ['locked'=>true,'retry_after'=>max(1, $ts - time())];
    }
  }

  // Count recent failures
  $st = $pdo->prepare("SELECT COUNT(*) FROM login_attempt_logs WHERE username=? AND ip=? AND success=0 AND attempted_at >= (NOW() - INTERVAL ? MINUTE)");
  $st->execute([$username, $ip, $windowMin]);
  $fails = (int)$st->fetchColumn();
  if ($fails >= $max) {
    // lock
    $pdo->prepare("INSERT INTO login_attempts (username, ip, locked_until) VALUES (?,?, (NOW() + INTERVAL ? MINUTE))
                   ON DUPLICATE KEY UPDATE locked_until=VALUES(locked_until)")
        ->execute([$username, $ip, $lockMin]);
    return ['locked'=>true,'retry_after'=>$lockMin*60];
  }

  return ['locked'=>false,'retry_after'=>0];
}

function mgz_login_log_attempt(PDO $pdo, string $username, string $ip, bool $success): void {
  $pdo->prepare("INSERT INTO login_attempt_logs (username, ip, success, attempted_at) VALUES (?,?,?,NOW())")
      ->execute([$username, $ip, $success ? 1 : 0]);
  if ($success) {
    // clear lock
    $pdo->prepare("DELETE FROM login_attempts WHERE username=? AND ip=?")
        ->execute([$username, $ip]);
  }
}
