<?php
/**
 * Security helpers: HTTP headers + safe redirects.
 */

function mgz_apply_security_headers(): void {
  static $done = false;
  if ($done) return;
  $done = true;

  // Avoid sending headers in CLI / already-sent cases.
  if (PHP_SAPI === 'cli' || headers_sent()) return;

  // Keep iframe shell working.
  header('X-Frame-Options: SAMEORIGIN');
  header('X-Content-Type-Options: nosniff');
  header('Referrer-Policy: strict-origin-when-cross-origin');

  // Conservative Permissions Policy (add features as needed).
  header('Permissions-Policy: geolocation=(), microphone=(), camera=()');

  // CSP: the project uses inline styles/scripts in many pages, so we allow unsafe-inline.
  // Tighten later when UI is refactored.
  $csp = implode('; ', [
    "default-src 'self'",
    "img-src 'self' data:",
    "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com",
    "font-src 'self' https://fonts.gstatic.com",
    "script-src 'self' 'unsafe-inline'",
    "frame-ancestors 'self'",
    "base-uri 'self'",
    "form-action 'self'",
  ]);
  header('Content-Security-Policy: ' . $csp);
}

/**
 * Allowlist internal redirects to prevent Open Redirect.
 *
 * Rules:
 * - Only relative paths.
 * - Must start with: "modules/" or one of known entry points.
 */
function mgz_is_safe_internal_path(string $path): bool {
  $path = trim($path);
  if ($path === '') return false;

  // Reject absolute URLs and protocol-relative URLs.
  if (preg_match('~^(https?:)?//~i', $path)) return false;

  // Normalize leading slash
  $path = ltrim($path);
  $allowed = [
    'dashboard.php',
    'index.php',
    'login.php',
    'logout.php',
  ];
  foreach ($allowed as $a) {
    if (str_starts_with($path, $a)) return true;
  }
  if (str_starts_with($path, 'modules/')) return true;
  return false;
}
