<?php
// Errors/Logging are handled centrally in app/db.php (and never displayed in production).

require_once __DIR__ . '/app/auth.php';
require_once __DIR__ . '/app/helpers.php';
require_once __DIR__ . '/app/db.php';

$u    = require_login();
$role = $u['role'];

$ROLE_AR = [
    'ADMIN'       => 'مدير النظام',
    'WAREHOUSE'   => 'أمين المخزن',
    'DISTRIBUTOR' => 'موزع',
];

$pdo     = db();
$stats   = [];
$top10 = [];
$lowStock = [];
$stockSnapshot = [];
$dbError = '';

/**
 * ملخص المخزون (يُستخدم للأدمن وأمين المخزن)
 * - إجمالي الوارد المعتمد
 * - إجمالي الصادر المعتمد للموزعين
 * - إجمالي المرتجع للمخزن
 * - الرصيد الكلي
 * - القيمة الإجمالية (من cover_price)
 */
$total_in               = 0;
$total_out_distributors = 0;
$total_returns          = 0;
$total_stock            = 0;
$total_value            = 0.0;

try {
    // إجمالي الوارد للمخزن (Ledger: WH_RECEIPT)
    $st = $pdo->query("
      SELECT COALESCE(SUM(qty_in),0) AS total_in
      FROM inventory_ledger
      WHERE status='POSTED'
        AND entity_type='WAREHOUSE' AND entity_id IS NULL
        AND ref_type='WH_RECEIPT'
    ");
    $total_in = (int)$st->fetchColumn();

    // إجمالي الصادر للموزعين (Ledger: DIST_ORDER / DISPATCH)
    $st = $pdo->query("
      SELECT COALESCE(SUM(qty_out),0) AS total_out_distributors
      FROM inventory_ledger
      WHERE status='POSTED'
        AND entity_type='WAREHOUSE' AND entity_id IS NULL
        AND ref_type IN ('DIST_ORDER','DISPATCH')
    ");
    $total_out_distributors = (int)$st->fetchColumn();

    // إجمالي المرتجع للمخزن (Ledger: WRT / WH_RETURN)
    $st = $pdo->query("
      SELECT COALESCE(SUM(qty_in),0) AS total_returns
      FROM inventory_ledger
      WHERE status='POSTED'
        AND entity_type='WAREHOUSE' AND entity_id IS NULL
        AND ref_type IN ('WRT','WH_RETURN')
    ");
    $total_returns = (int)$st->fetchColumn();

    // الرصيد الكلي = الوارد المعتمد − الصادر المعتمد + المرتجع المعتمد
    $total_stock = $total_in - $total_out_distributors + $total_returns;

    // حساب القيمة الإجمالية للمخزون من جدول الإصدارات كما في تقرير المخزون
    $rowsVal = $pdo->query("
        SELECT i.id, i.cover_price, COALESCE(v.balance_qty,0) AS warehouse_balance
        FROM issues i
        LEFT JOIN v_inventory_balance v
          ON v.entity_type='WAREHOUSE' AND v.entity_id IS NULL AND v.issue_id=i.id
    ")->fetchAll();

    $total_value = array_sum(array_map(
        fn($r) => (int)$r['warehouse_balance'] * (float)$r['cover_price'],
        $rowsVal
    ));

    // إحصائيات حسب الدور (Dashboard Summary)
    if ($role === 'ADMIN') {

        // Pending Queue
        $stats = $pdo->query("
            SELECT
                (SELECT COUNT(*) FROM warehouse_receipts WHERE status='SUBMITTED')              AS whr_pending,
                (SELECT COUNT(*) FROM distributor_dispatches WHERE status='SUBMITTED')          AS dispatch_pending,
                (SELECT COUNT(*) FROM damage_reports WHERE status='PENDING')                    AS damage_pending,
                (SELECT COUNT(*) FROM outlet_settlements WHERE status='SUBMITTED')              AS settlements_pending,
                (SELECT COUNT(*) FROM invoices WHERE status IN ('ISSUED','PARTIAL'))            AS open_invoices,
                (SELECT COALESCE(SUM(balance_due),0) FROM invoices WHERE status IN ('ISSUED','PARTIAL')) AS total_due
        ")->fetch();

        // Top 10 Distributors by due
        $top10 = $pdo->query("
            SELECT d.id, d.name, COALESCE(SUM(i.balance_due),0) AS due
            FROM distributors d
            LEFT JOIN invoices i
              ON i.distributor_id = d.id
             AND i.status IN ('ISSUED','PARTIAL')
            GROUP BY d.id, d.name
            HAVING COALESCE(SUM(i.balance_due),0) > 0
            ORDER BY due DESC
            LIMIT 10
        ")->fetchAll();

        // Stock snapshot
        $stAvail = $pdo->query("SELECT COALESCE(SUM(balance_qty),0) FROM v_inventory_balance WHERE entity_type='WAREHOUSE' AND entity_id IS NULL")->fetchColumn();
        $stHold  = $pdo->query("SELECT COALESCE(SUM(qty_in-qty_out),0) FROM inventory_ledger WHERE entity_type='WAREHOUSE' AND entity_id IS NULL AND bucket='HOLD' AND status='POSTED'")->fetchColumn();
        $stDam   = $pdo->query("SELECT COALESCE(SUM(balance_qty),0) FROM v_inventory_balance WHERE entity_type='DAMAGED'")->fetchColumn();

        // InTransit (لو موجود): كميات في dispatches المرسلة ولم تُعتمد بعد
        $stInTransit = 0;
        try {
            $stInTransit = (int)$pdo->query("
                SELECT COALESCE(SUM(di.qty),0)
                FROM distributor_dispatches dd
                JOIN distributor_dispatch_items di ON di.dispatch_id = dd.id
                WHERE dd.status='SUBMITTED'
            ")->fetchColumn();
        } catch (Throwable $e2) {
            $stInTransit = 0;
        }

        $stockSnapshot = [
            'available' => (int)$stAvail,
            'hold'      => (int)$stHold,
            'damaged'   => (int)$stDam,
            'intransit' => (int)$stInTransit,
        ];

        // Low stock list (أقل 10 أصناف بالمخزن)
        $lowStock = $pdo->query("
            SELECT p.name AS product_name, i.issue_month, i.issue_year, i.issue_number, s.balance
            FROM v_admin_stock_summary s
            JOIN issues i ON i.id = s.issue_id
            JOIN products p ON p.id = i.product_id
            ORDER BY s.balance ASC
            LIMIT 10
        ")->fetchAll();

    } elseif ($role === 'WAREHOUSE') {

        $stats = $pdo->query("
            SELECT
                (SELECT COUNT(*) FROM warehouse_receipts WHERE status='SUBMITTED') AS whr_pending,
                (SELECT COUNT(*) FROM warehouse_receipts WHERE status='APPROVED' AND manager_at IS NOT NULL AND DATE(manager_at)=CURDATE()) AS whr_approved_today,
                (SELECT COUNT(*) FROM distributor_dispatches WHERE status='SUBMITTED') AS dispatch_pending
        ")->fetch();

        // Stock snapshot (للمخزن)
        $stAvail = $pdo->query("SELECT COALESCE(SUM(balance_qty),0) FROM v_inventory_balance WHERE entity_type='WAREHOUSE' AND entity_id IS NULL")->fetchColumn();
        $stHold  = $pdo->query("SELECT COALESCE(SUM(qty_in-qty_out),0) FROM inventory_ledger WHERE entity_type='WAREHOUSE' AND entity_id IS NULL AND bucket='HOLD' AND status='POSTED'")->fetchColumn();
        $stDam   = $pdo->query("SELECT COALESCE(SUM(balance_qty),0) FROM v_inventory_balance WHERE entity_type='DAMAGED'")->fetchColumn();

        $stockSnapshot = [
            'available' => (int)$stAvail,
            'hold'      => (int)$stHold,
            'damaged'   => (int)$stDam,
            'intransit' => (int)($stats['dispatch_pending'] ?? 0), // هنا كعدد طلبات (مش كمية)
        ];

        // Low stock alerts
        $lowStock = $pdo->query("
            SELECT p.name AS product_name, i.issue_month, i.issue_year, i.issue_number, s.balance
            FROM v_admin_stock_summary s
            JOIN issues i ON i.id = s.issue_id
            JOIN products p ON p.id = i.product_id
            ORDER BY s.balance ASC
            LIMIT 10
        ")->fetchAll();

    } elseif ($role === 'DISTRIBUTOR') {

        $distId = (int)($u['distributor_id'] ?? 0);

        $st2 = $pdo->prepare("
            SELECT
                (SELECT COUNT(*) FROM invoices
                 WHERE distributor_id=?
                   AND status IN ('ISSUED','PARTIAL'))                                  AS open_invoices,
                (SELECT COALESCE(SUM(balance_due),0) FROM invoices
                 WHERE distributor_id=?
                   AND status IN ('ISSUED','PARTIAL'))                                  AS total_due,
                (SELECT COUNT(*) FROM outlet_settlements
                 WHERE distributor_id=? AND status='SUBMITTED')                          AS settlements_pending,
                (SELECT COUNT(*) FROM outlet_settlements
                 WHERE distributor_id=? AND status='APPROVED')                           AS settlements_approved,
                (SELECT COALESCE(SUM(amount),0) FROM payments
                 WHERE distributor_id=? AND status='HELD')                               AS payments_held,
                (SELECT COALESCE(SUM(amount),0) FROM payments
                 WHERE distributor_id=? AND status='POSTED')                             AS payments_posted,
                (SELECT COALESCE(SUM(qty_in-qty_out),0) FROM inventory_ledger
                 WHERE entity_type='DISTRIBUTOR' AND entity_id=? AND bucket='HOLD' AND status='POSTED') AS hold_qty
        ");
        $st2->execute([$distId, $distId, $distId, $distId, $distId, $distId, $distId]);
        $stats = $st2->fetch();
    }
} catch (Throwable $e) {
    $stats   = [];
    $dbError = $e->getMessage();
}

// (تم حذف صندوق "المتوفر بالمخزن (أحدث 5)" بناءً على طلب العميل)

?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>لوحة التحكم — MGZ</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;500;600;700&display=swap" rel="stylesheet">
<style>
*, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }
body {
    font-family: 'Tajawal', sans-serif;
    background: #f0f2f5;
    direction: rtl;
    min-height: 100vh;
}
.topbar {
    background: linear-gradient(135deg, #1a73e8, #0d47a1);
    color: #fff;
    padding: 0 24px;
    height: 60px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    position: sticky;
    top: 0;
    z-index: 100;
    box-shadow: 0 2px 8px rgba(0,0,0,.15);
}
.topbar .brand { font-size: 1.2rem; font-weight: 700; }
.topbar .brand span {
    background: rgba(255,255,255,.2);
    padding: 2px 10px;
    border-radius: 6px;
    margin-right: 8px;
    font-size: .85rem;
}
.topbar .user-info { display: flex; align-items: center; gap: 14px; font-size: .9rem; }
.topbar .role-badge {
    background: rgba(255,255,255,.2);
    padding: 3px 12px;
    border-radius: 20px;
    font-size: .8rem;
}
.topbar .logout-btn {
    background: rgba(255,255,255,.15);
    border: 1px solid rgba(255,255,255,.3);
    color: #fff;
    padding: 5px 14px;
    border-radius: 6px;
    text-decoration: none;
    font-size: .85rem;
}
.topbar .logout-btn:hover { background: rgba(255,255,255,.25); }
.main { padding: 24px; max-width: 1200px; margin: auto; }
.welcome-banner {
    background: linear-gradient(135deg, #1a73e8, #6610f2);
    color: #fff;
    padding: 20px 28px;
    border-radius: 12px;
    margin-bottom: 24px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 12px;
}
.welcome-banner h2 { font-size: 1.3rem; font-weight: 700; margin-bottom: 4px; }
.welcome-banner p  { opacity: .85; font-size: .9rem; }
.welcome-banner .date-box {
    background: rgba(255,255,255,.15);
    padding: 8px 16px;
    border-radius: 8px;
    font-size: .88rem;
    text-align: center;
}
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(160px,1fr));
    gap: 14px;
    margin-bottom: 28px;
}
.stat-card {
    background: #fff;
    border-radius: 12px;
    padding: 18px 16px;
    box-shadow: 0 2px 8px rgba(0,0,0,.07);
    text-align: center;
    border-top: 4px solid transparent;
    transition: transform .2s;
}
.stat-card:hover { transform: translateY(-3px); }
.stat-card.blue   { border-color: #1a73e8; }
.stat-card.green  { border-color: #34a853; }
.stat-card.orange { border-color: #fbbc04; }
.stat-card.red    { border-color: #ea4335; }
.stat-card.purple { border-color: #6f42c1; }
.stat-card .num   { font-size: 1.9rem; font-weight: 700; line-height: 1; margin-bottom: 6px; }
.stat-card .lbl   { font-size: .8rem; color: #888; }
.stat-card.blue   .num { color: #1a73e8; }
.stat-card.green  .num { color: #34a853; }
.stat-card.orange .num { color: #f57c00; }
.stat-card.red    .num { color: #ea4335; }
.stat-card.purple .num { color: #6f42c1; }

.inventory-summary {
    display:flex;
    flex-wrap:wrap;
    gap:15px;
    margin-bottom:24px;
}
.inv-card {
    flex:1 1 180px;
    border-radius:10px;
    padding:14px 18px;
    box-shadow:0 2px 8px rgba(15,23,42,.15);
    color:#fff;
}
.inv-card .lbl {
    font-size:.8rem;
    opacity:.9;
}
.inv-card .num {
    font-size:1.6rem;
    font-weight:700;
    margin-top:4px;
}
.inv-main {
    flex:1 1 220px;
    border-radius:10px;
    padding:14px 18px;
    box-shadow:0 3px 10px rgba(15,23,42,.35);
    border:2px solid #22c55e;
    background:#111827;
    color:#f9fafb;
}
.inv-main .lbl {
    font-size:.8rem;
    opacity:.9;
}
.inv-main .num {
    font-size:1.8rem;
    font-weight:800;
    margin-top:4px;
}
.inv-main .hint {
    font-size:.8rem;
    color:#a7f3d0;
    margin-top:3px;
}

.section-title {
    font-size: 1rem;
    font-weight: 700;
    color: #555;
    margin-bottom: 12px;
    margin-top: 8px;
    display: flex;
    align-items: center;
    gap: 8px;
}
.section-title::after { content:''; flex:1; height:1px; background:#e0e0e0; }
.menu-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(170px,1fr));
    gap: 12px;
    margin-bottom: 24px;
}
.menu-card {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    gap: 8px;
    padding: 20px 14px;
    background: #fff;
    border-radius: 12px;
    border: 2px solid #eee;
    text-decoration: none;
    color: #444;
    font-size: .88rem;
    font-weight: 600;
    text-align: center;
    transition: all .2s;
    box-shadow: 0 1px 4px rgba(0,0,0,.05);
}
.menu-card:hover {
    border-color: #1a73e8;
    background: #e8f0fe;
    color: #1a73e8;
    transform: translateY(-2px);
}
.menu-card .icon { font-size: 1.8rem; line-height: 1; }
.menu-card .code {
    font-size: .72rem;
    background: #f1f3f4;
    color: #888;
    padding: 1px 8px;
    border-radius: 10px;
    font-weight: 400;
}
.menu-card:hover .code { background: #c5d9fb; color: #1a73e8; }

/* Notifications (compact, scrollable) */
.notif-panel{
    background:#fff;
    border:2px solid #eee;
    border-radius:14px;
    box-shadow: 0 1px 4px rgba(0,0,0,.05);
    overflow:auto;
    max-height: 260px;
    margin-bottom: 18px;
}
.notif-item{
    display:flex;
    gap:10px;
    padding:12px 12px;
    text-decoration:none;
    color:#111827;
    border-bottom:1px solid #f1f3f4;
    align-items:flex-start;
}
.notif-item:last-child{border-bottom:none;}
.notif-item:hover{background:#f8fafc;}
.notif-ico{font-size:1.2rem; line-height:1.2;}
.notif-txt{display:flex; flex-direction:column; gap:4px; min-width:0;}
.notif-title{font-weight:800; font-size:.95rem; line-height:1.25;}
.notif-body{font-size:.85rem; color:#4b5563; line-height:1.35; word-break:break-word;}
.notif-time{font-size:.75rem; color:#9ca3af;}
.alert-danger {
    background: #fce8e6;
    color: #c62828;
    border-right: 4px solid #ea4335;
    padding: 14px 18px;
    border-radius: 8px;
    margin-bottom: 20px;
    font-size: .9rem;
}

.mini-stock{
    background:#fff;
    border-radius:14px;
    padding:12px 12px;
    border:1px solid rgba(0,0,0,.06);
    box-shadow:0 6px 18px rgba(0,0,0,.06);
    min-width: 260px;
}
.mini-stock h4{font-size:.95rem;margin-bottom:8px;color:#111827;}
.mini-stock table{width:100%;border-collapse:collapse;font-size:.85rem;}
.mini-stock td{padding:6px 0;border-bottom:1px dashed #eee;vertical-align:top;}
.mini-stock td.qty{font-weight:800;color:#0d47a1;text-align:left;white-space:nowrap;}
.mini-stock a.more{display:inline-block;margin-top:8px;font-size:.82rem;color:#1a73e8;text-decoration:none;font-weight:700;}
@media (max-width: 600px) {
    .main { padding: 14px; }
    .stats-grid { grid-template-columns: repeat(2,1fr); }
    .menu-grid  { grid-template-columns: repeat(2,1fr); }
    .welcome-banner { flex-direction: column; }
    .topbar .brand span { display: none; }
}

.section-title{
    margin:18px 0 10px;
    font-size:1.05rem;
    font-weight:700;
    color:#0f172a;
}
.muted{ color:#64748b; font-size:.95rem; }

.table-wrap{
    background:#fff;
    border:1px solid #e5e7eb;
    border-radius:14px;
    padding:12px;
}
.table-title{
    font-weight:700;
    margin-bottom:10px;
    color:#111827;
}
.tbl{
    width:100%;
    border-collapse:collapse;
}
.tbl th, .tbl td{
    padding:10px 8px;
    border-bottom:1px solid #eef2f7;
    font-size:.95rem;
}
.tbl thead th{
    background:#f8fafc;
    color:#0f172a;
}
</style>

<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="assets/ui.css">
<script defer src="assets/ui.js"></script>
</head>
<body>

<div class="topbar">
    <div class="brand">
        &#128240; MGZ
        <span>نظام توزيع المجلات</span>
    </div>
    <div class="user-info">
        <span>&#128100; <?= h($u['name']) ?></span>
        <span class="role-badge"><?= $ROLE_AR[$role] ?? $role ?></span>
        <a href="logout.php" class="logout-btn">&#128275; خروج</a>
    </div>
</div>

<div class="main">

    <?php if ($dbError !== ''): ?>
    <div class="alert-danger">
        <strong>&#9888; خطأ في قاعدة البيانات:</strong><br>
        <?= htmlspecialchars($dbError) ?>
    </div>
    <?php endif; ?>

    <div class="welcome-banner">
        <div>
            <h2>أهلاً بك، <?= h($u['name']) ?> &#128075;</h2>
            <p><?= $ROLE_AR[$role] ?? $role ?> — نظام إدارة توزيع المجلات</p>
        </div>
        <div class="date-box">
            <div style="font-size:1.1rem;font-weight:700;"><?= date('d/m/Y') ?></div>
            <div style="opacity:.8;font-size:.82rem;"><?= date('l') ?></div>
        </div>
    <!-- (تم حذف نظام الإشعارات بناءً على طلب العميل) -->
    </div>

    <?php if (in_array($role, ['ADMIN','WAREHOUSE'], true)): ?>
    <!-- ملخص رصيد المخزون في أعلى الصفحة -->
    <div class="inventory-summary">
        <div class="inv-card" style="background:#0f766e;">
            <div class="lbl">إجمالي الوارد للمخزن</div>
            <div class="num"><?= fmt_int($total_in) ?></div>
        </div>
        <div class="inv-card" style="background:#b45309;">
            <div class="lbl">إجمالي الصادر للموزعين</div>
            <div class="num"><?= fmt_int($total_out_distributors) ?></div>
        </div>
        <div class="inv-card" style="background:#0369a1;">
            <div class="lbl">إجمالي المرتجع للمخزن</div>
            <div class="num"><?= fmt_int($total_returns) ?></div>
        </div>
        <div class="inv-main">
            <div class="lbl">إجمالي الإصدارات بالخزن (الرصيد الكلي)</div>
            <div class="num"><?= fmt_int($total_stock) ?></div>
            <div class="hint">(الوارد المعتمد − الصادر المعتمد + المرتجع)</div>
        </div>
        <div class="inv-card" style="background:#f97316;">
            <div class="lbl">القيمة الإجمالية (ر.ع.)</div>
            <div class="num"><?= omr($total_value) ?></div>
        </div>
    </div>
    <?php endif; ?>

    
    <?php if ($role === 'ADMIN'): ?>

        <h2 class="section-title">Pending Queue</h2>
        <div class="stats-grid">
            <div class="stat-card purple">
                <div class="num"><?= (int)($stats['whr_pending'] ?? 0) ?></div>
                <div class="lbl">WHR بانتظار الاعتماد</div>
            </div>
            <div class="stat-card blue">
                <div class="num"><?= (int)($stats['dispatch_pending'] ?? 0) ?></div>
                <div class="lbl">طلبات الموزعين (Workflow)</div>
            </div>
            <div class="stat-card orange">
                <div class="num"><?= (int)($stats['damage_pending'] ?? 0) ?></div>
                <div class="lbl">طلبات التالف (اعتماد المدير)</div>
            </div>
            <div class="stat-card orange">
                <div class="num"><?= (int)($stats['settlements_pending'] ?? 0) ?></div>
                <div class="lbl">تسويات أمانة معلّقة</div>
            </div>
            <div class="stat-card red">
                <div class="num"><?= (int)($stats['open_invoices'] ?? 0) ?></div>
                <div class="lbl">فواتير مفتوحة</div>
            </div>
        </div>

        <h2 class="section-title">Money &amp; Risk</h2>
        <div class="stats-grid">
            <div class="stat-card red">
                <div class="num" style="font-size:1.2rem;"><?= omr((float)($stats['total_due'] ?? 0)) ?></div>
                <div class="lbl">إجمالي المستحقات (Open invoices)</div>
            </div>
            <div class="stat-card blue">
                <div class="num"><?= (int)($stats['open_invoices'] ?? 0) ?></div>
                <div class="lbl">عدد الفواتير المفتوحة</div>
            </div>
        </div>

        <?php if (!empty($top10)): ?>
            <div class="table-wrap" style="margin-top:12px;">
                <div class="table-title">Top 10 موزعين حسب مديونيات منافذهم</div>
                <table class="tbl">
                    <thead>
                        <tr>
                            <th>الموزع</th>
                            <th style="text-align:left;">إجمالي المديونية (ر.ع.)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($top10 as $r): ?>
                        <tr>
                            <td><?= h($r['name']) ?></td>
                            <td style="text-align:left;"><?= omr((float)$r['due']) ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="muted" style="margin-top:10px;">لا توجد مديونيات على الموزعين حالياً.</div>
        <?php endif; ?>

        <h2 class="section-title">Stock snapshot</h2>
        <div class="stats-grid">
            <div class="stat-card green">
                <div class="num"><?= (int)($stockSnapshot['available'] ?? 0) ?></div>
                <div class="lbl">Available (Warehouse)</div>
            </div>
            <div class="stat-card purple">
                <div class="num"><?= (int)($stockSnapshot['hold'] ?? 0) ?></div>
                <div class="lbl">Hold (Warehouse)</div>
            </div>
            <div class="stat-card orange">
                <div class="num"><?= (int)($stockSnapshot['damaged'] ?? 0) ?></div>
                <div class="lbl">Damaged</div>
            </div>
            <div class="stat-card blue">
                <div class="num"><?= (int)($stockSnapshot['intransit'] ?? 0) ?></div>
                <div class="lbl">InTransit</div>
            </div>
        </div>

        <?php if (!empty($lowStock)): ?>
            <div class="table-wrap" style="margin-top:12px;">
                <div class="table-title">Low stock alerts (أقل 10 أصناف بالمخزن)</div>
                <table class="tbl">
                    <thead>
                        <tr>
                            <th>المنتج</th>
                            <th style="text-align:center;">الإصدار</th>
                            <th style="text-align:left;">الرصيد</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($lowStock as $r): ?>
                        <tr>
                            <td><?= h($r['product_name']) ?></td>
                            <td style="text-align:center;">
                                <?= (int)$r['issue_month'] ?>/<?= (int)$r['issue_year'] ?>
                                <?php if (!empty($r['issue_number'])): ?> (<?= (int)$r['issue_number'] ?>)<?php endif; ?>
                            </td>
                            <td style="text-align:left;"><?= (int)$r['balance'] ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

    <?php elseif ($role === 'DISTRIBUTOR'): ?>

        <h2 class="section-title">ملخص الحساب</h2>
        <div class="stats-grid">
            <div class="stat-card blue">
                <div class="num"><?= (int)($stats['open_invoices'] ?? 0) ?></div>
                <div class="lbl">فواتير مفتوحة</div>
            </div>
            <div class="stat-card red">
                <div class="num" style="font-size:1.2rem;"><?= omr((float)($stats['total_due'] ?? 0)) ?></div>
                <div class="lbl">إجمالي المستحق (ر.ع.)</div>
            </div>
        </div>

        <h2 class="section-title">تسويات الأمانة</h2>
        <div class="stats-grid">
            <div class="stat-card orange">
                <div class="num"><?= (int)($stats['settlements_pending'] ?? 0) ?></div>
                <div class="lbl">Pending</div>
            </div>
            <div class="stat-card green">
                <div class="num"><?= (int)($stats['settlements_approved'] ?? 0) ?></div>
                <div class="lbl">Approved</div>
            </div>
        </div>

        <h2 class="section-title">التحصيلات</h2>
        <div class="stats-grid">
            <div class="stat-card orange">
                <div class="num" style="font-size:1.2rem;"><?= omr((float)($stats['payments_held'] ?? 0)) ?></div>
                <div class="lbl">Held</div>
            </div>
            <div class="stat-card green">
                <div class="num" style="font-size:1.2rem;"><?= omr((float)($stats['payments_posted'] ?? 0)) ?></div>
                <div class="lbl">Posted</div>
            </div>
        </div>

        <h2 class="section-title">مخزون HOLD عليك</h2>
        <div class="stats-grid">
            <div class="stat-card purple">
                <div class="num"><?= (int)($stats['hold_qty'] ?? 0) ?></div>
                <div class="lbl">HOLD Qty</div>
            </div>
        </div>

    <?php elseif ($role === 'WAREHOUSE'): ?>

        <h2 class="section-title">ملخص المخزن</h2>
        <div class="stats-grid">
            <div class="stat-card purple">
                <div class="num"><?= (int)($stats['whr_pending'] ?? 0) ?></div>
                <div class="lbl">WHR pending</div>
            </div>
            <div class="stat-card green">
                <div class="num"><?= (int)($stats['whr_approved_today'] ?? 0) ?></div>
                <div class="lbl">WHR approved today</div>
            </div>
            <div class="stat-card blue">
                <div class="num"><?= (int)($stats['dispatch_pending'] ?? 0) ?></div>
                <div class="lbl">Dispatches to be picked</div>
            </div>
        </div>

        <h2 class="section-title">Stock snapshot</h2>
        <div class="stats-grid">
            <div class="stat-card green">
                <div class="num"><?= (int)($stockSnapshot['available'] ?? 0) ?></div>
                <div class="lbl">Available</div>
            </div>
            <div class="stat-card purple">
                <div class="num"><?= (int)($stockSnapshot['hold'] ?? 0) ?></div>
                <div class="lbl">Hold</div>
            </div>
            <div class="stat-card orange">
                <div class="num"><?= (int)($stockSnapshot['damaged'] ?? 0) ?></div>
                <div class="lbl">Damaged</div>
            </div>
        </div>

        <?php if (!empty($lowStock)): ?>
            <div class="table-wrap" style="margin-top:12px;">
                <div class="table-title">Low stock alerts</div>
                <table class="tbl">
                    <thead>
                        <tr>
                            <th>المنتج</th>
                            <th style="text-align:center;">الإصدار</th>
                            <th style="text-align:left;">الرصيد</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($lowStock as $r): ?>
                        <tr>
                            <td><?= h($r['product_name']) ?></td>
                            <td style="text-align:center;">
                                <?= (int)$r['issue_month'] ?>/<?= (int)$r['issue_year'] ?>
                                <?php if (!empty($r['issue_number'])): ?> (<?= (int)$r['issue_number'] ?>)<?php endif; ?>
                            </td>
                            <td style="text-align:left;"><?= (int)$r['balance'] ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

    <?php endif; ?>


    <?php if ($role === 'ADMIN'): ?>
    <div class="section-title">&#128218; البيانات الأساسية</div>
    <div class="menu-grid">
        <a href="modules/products.php" class="menu-card">
            <span class="icon">&#128240;</span>
            <span>المنتجات</span>
        </a>
	        <a href="modules/issues.php" class="menu-card">
	            <span class="icon">📚</span>
	            <span>الإصدارات</span>
	        </a>
<a href="modules/outlets.php" class="menu-card">
            <span class="icon">&#127978;</span><span>منافذ البيع</span>
        </a>
        <a href="modules/distributors.php" class="menu-card">
            <span class="icon">&#128666;</span><span>الموزعون</span>
        </a>
    </div>

    <div class="section-title">&#128203; المستندات والحركات</div>
    <div class="menu-grid">
        <a href="modules/wh_receipts.php" class="menu-card">
            <span class="icon">&#128230;</span>
            <span>توريد المخزن</span>
            <span class="code">WHR</span>
        </a>
        <a href="modules/distributor_orders.php" class="menu-card">
            <span class="icon">🧾</span>
            <span>طلبات الموزعين</span>
            <span class="code">DOR</span>
        </a>
<a href="modules/warehouse_returns.php" class="menu-card">
            <span class="icon">&#128257;</span>
            <span>مرتجع للمخزن</span>
            <span class="code">WRT</span>
        </a>
                <a href="modules/damage_reports.php" class="menu-card">
            <span class="icon">🧾</span>
            <span>طلبات التالف</span>
            <span class="code">DMG</span>
        </a>
<a href="modules/outlet_deliveries.php" class="menu-card">
            <span class="icon">&#128667;</span>
            <span>تسليم للمنافذ</span>
            <span class="code">ODL</span>
        </a>
        <a href="modules/settlements.php" class="menu-card">
            <span class="icon">&#128203;</span>
            <span>تسويات الأمانة</span>
            <span class="code">OST</span>
        </a>
        <a href="modules/invoices.php" class="menu-card">
            <span class="icon">&#128196;</span>
            <span>الفواتير</span>
            <span class="code">INV</span>
        </a>
        <a href="modules/payments.php" class="menu-card">
            <span class="icon">&#128178;</span>
            <span>التحصيلات</span>
            <span class="code">PAY</span>
        </a>
    </div>

    <div class="section-title">&#128200; التقارير</div>
    <div class="menu-grid">
        <a href="modules/report_stock.php" class="menu-card">
            <span class="icon">&#128218;</span>
            <span>تقرير المخزون</span>
        </a>
        <a href="modules/report_issue_summary.php" class="menu-card">
            <span class="icon">📌</span>
            <span>ملخص الإصدار</span>
        </a>
        <a href="modules/report_outlet_statement.php" class="menu-card">
            <span class="icon">&#128196;</span>
            <span>كشف حساب منفذ</span>
        </a>
        <a href="modules/report_outlet_arrears.php" class="menu-card">
            <span class="icon">🧾</span>
            <span>مديونيات المنافذ</span>
        </a>
        <a href="modules/report_distributor_holdings.php" class="menu-card">
            <span class="icon">&#128666;</span>
            <span>عهدة الموزعين</span>
        </a>
        <a href="modules/report_distributor_commission.php" class="menu-card">
            <span class="icon">&#128200;</span>
            <span>عمولات الموزعين</span>
        </a>
        <a href="modules/report_office_collections.php" class="menu-card">
            <span class="icon">💰</span>
            <span>تحصيلات الموزعين للمكتب</span>
        </a>
        <a href="modules/report_ledger.php" class="menu-card">
            <span class="icon">📒</span>
            <span>دفتر الأستاذ</span>
        </a>
    </div>

    <?php elseif ($role === 'WAREHOUSE'): ?>

    <div class="section-title">&#128218; عمليات المخزن</div>
    <div class="menu-grid">
<a href="modules/wh_receipts.php" class="menu-card">
            <span class="icon">&#128230;</span>
            <span>توريد المخزن</span>
            <span class="code">WHR</span>
        </a>
        <a href="modules/distributor_orders.php" class="menu-card">
            <span class="icon">🧾</span>
            <span>طلبات الموزعين</span>
            <span class="code">DOR</span>
        </a>
<a href="modules/warehouse_returns.php" class="menu-card">
            <span class="icon">&#128257;</span>
            <span>مرتجعات للمخزن</span>
            <span class="code">WRT</span>
        </a>
        <a href="modules/report_stock.php" class="menu-card">
            <span class="icon">&#128218;</span>
            <span>تقرير المخزون</span>
        </a>
    </div>

    <?php elseif ($role === 'DISTRIBUTOR'): ?>

    <div class="section-title">🚚 عمليات الموزع</div>
    <div class="menu-grid">
        <a href="modules/distributor_orders.php" class="menu-card">
            <span class="icon">🧾</span>
            <span>طلب صرف من المخزن</span>
            <span class="code">DOR</span>
        </a>
        <a href="modules/outlets.php" class="menu-card">
            <span class="icon">🏪</span>
            <span>منافذ البيع الخاصة بي</span>
            <span class="code">OUT</span>
        </a>
        <a href="modules/outlet_deliveries.php" class="menu-card">
            <span class="icon">🚛</span>
            <span>تسليم للمنافذ</span>
            <span class="code">ODL</span>
        </a>
        <a href="modules/settlements.php" class="menu-card">
            <span class="icon">🧮</span>
            <span>تسوية (بيع/مرتجع/تالف)</span>
            <span class="code">OST</span>
        </a>
        <a href="modules/invoices.php" class="menu-card">
            <span class="icon">🧾</span>
            <span>الفواتير</span>
            <span class="code">INV</span>
        </a>
        <a href="modules/distributor_receivables.php" class="menu-card">
            <span class="icon">💰</span>
            <span>مستحقات المنافذ</span>
            <span class="code">RCV</span>
        </a>
</div>

    <?php endif; ?>

    <div style="text-align:center;color:#bbb;font-size:.8rem;margin-top:30px;padding-top:16px;border-top:1px solid #eee;">
        MGZ — نظام إدارة توزيع المجلات &copy; <?= date('Y') ?>
    </div>

</div>
</body>
</html>
