-- MGZ Patch: security + integrity (Idempotent)
-- Apply on existing DB (MySQL/MariaDB).
-- Safe to run multiple times.

START TRANSACTION;

-- =====================================================
-- Helpers: dynamic add column / index / unique, safely
-- =====================================================

-- 1) Attachments: add ownership / entity linkage for RBAC downloads
-- attachments.entity_type
SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'attachments'
    AND column_name = 'entity_type'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE attachments ADD COLUMN entity_type VARCHAR(40) NULL AFTER original_name',
  'SELECT ''attachments.entity_type exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- attachments.entity_id
SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'attachments'
    AND column_name = 'entity_id'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE attachments ADD COLUMN entity_id BIGINT(20) UNSIGNED NULL AFTER entity_type',
  'SELECT ''attachments.entity_id exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- attachments.distributor_id
SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'attachments'
    AND column_name = 'distributor_id'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE attachments ADD COLUMN distributor_id BIGINT(20) UNSIGNED NULL AFTER entity_id',
  'SELECT ''attachments.distributor_id exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- attachments.outlet_id
SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'attachments'
    AND column_name = 'outlet_id'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE attachments ADD COLUMN outlet_id BIGINT(20) UNSIGNED NULL AFTER distributor_id',
  'SELECT ''attachments.outlet_id exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- Index ix_att_entity (entity_type, entity_id)
SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'attachments'
    AND index_name = 'ix_att_entity'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE attachments ADD KEY ix_att_entity (entity_type, entity_id)',
  'SELECT ''attachments.ix_att_entity exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- Index ix_att_distributor (distributor_id)
SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'attachments'
    AND index_name = 'ix_att_distributor'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE attachments ADD KEY ix_att_distributor (distributor_id)',
  'SELECT ''attachments.ix_att_distributor exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- Index ix_att_outlet (outlet_id)
SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'attachments'
    AND index_name = 'ix_att_outlet'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE attachments ADD KEY ix_att_outlet (outlet_id)',
  'SELECT ''attachments.ix_att_outlet exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;


-- =====================================================
-- 2) Uniques to prevent duplicate documents / allocations
-- =====================================================

-- outlet_settlements UNIQUE uk_outlet_settlement_delivery (delivery_id)
SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'outlet_settlements'
    AND index_name = 'uk_outlet_settlement_delivery'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE outlet_settlements ADD UNIQUE KEY uk_outlet_settlement_delivery (delivery_id)',
  'SELECT ''outlet_settlements.uk_outlet_settlement_delivery exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- invoices UNIQUE uk_invoice_source (source_type, source_id)
SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'invoices'
    AND index_name = 'uk_invoice_source'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE invoices ADD UNIQUE KEY uk_invoice_source (source_type, source_id)',
  'SELECT ''invoices.uk_invoice_source exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- payment_allocations UNIQUE uk_payment_invoice (payment_id, invoice_id)
SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'payment_allocations'
    AND index_name = 'uk_payment_invoice'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE payment_allocations ADD UNIQUE KEY uk_payment_invoice (payment_id, invoice_id)',
  'SELECT ''payment_allocations.uk_payment_invoice exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;


-- =====================================================
-- 3) Brute force protection tables (compatible)
-- =====================================================

-- Create login_attempt_logs if missing
CREATE TABLE IF NOT EXISTS login_attempt_logs (
  id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
  username VARCHAR(190) NOT NULL,
  ip VARCHAR(64) NOT NULL,
  success TINYINT(1) NOT NULL DEFAULT 0,
  attempted_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY ix_login_logs_user_ip_time (username, ip, attempted_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Ensure attempted_at exists (if table existed older variant)
SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'login_attempt_logs'
    AND column_name = 'attempted_at'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE login_attempt_logs ADD COLUMN attempted_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP',
  'SELECT ''login_attempt_logs.attempted_at exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- Ensure index on attempted_at (your runtime fix)
SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'login_attempt_logs'
    AND index_name = 'idx_login_attempt_logs_attempted'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE login_attempt_logs ADD KEY idx_login_attempt_logs_attempted (attempted_at)',
  'SELECT ''login_attempt_logs.idx_login_attempt_logs_attempted exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- Create login_attempts if missing (use structure compatible with throttle code)
CREATE TABLE IF NOT EXISTS login_attempts (
  username VARCHAR(190) NOT NULL,
  ip VARCHAR(64) NOT NULL,
  locked_until DATETIME NOT NULL,
  PRIMARY KEY (username, ip)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Optional extra columns used by some versions of throttle (safe add)
-- fail_count
SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'login_attempts'
    AND column_name = 'fail_count'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE login_attempts ADD COLUMN fail_count INT UNSIGNED NOT NULL DEFAULT 0 AFTER ip',
  'SELECT ''login_attempts.fail_count exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

-- last_fail_at
SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'login_attempts'
    AND column_name = 'last_fail_at'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE login_attempts ADD COLUMN last_fail_at DATETIME NULL AFTER locked_until',
  'SELECT ''login_attempts.last_fail_at exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;


-- =====================================================
-- 4) Optional: add attachment id to damage_reports for future normalization
-- =====================================================

SET @col_exists := (
  SELECT COUNT(*)
  FROM information_schema.COLUMNS
  WHERE table_schema = DATABASE()
    AND table_name = 'damage_reports'
    AND column_name = 'photo_attachment_id'
);
SET @sql := IF(@col_exists = 0,
  'ALTER TABLE damage_reports ADD COLUMN photo_attachment_id BIGINT(20) UNSIGNED NULL AFTER photo_path',
  'SELECT ''damage_reports.photo_attachment_id exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;

SET @idx_exists := (
  SELECT COUNT(*)
  FROM information_schema.statistics
  WHERE table_schema = DATABASE()
    AND table_name = 'damage_reports'
    AND index_name = 'ix_damage_photo_att'
);
SET @sql := IF(@idx_exists = 0,
  'ALTER TABLE damage_reports ADD KEY ix_damage_photo_att (photo_attachment_id)',
  'SELECT ''damage_reports.ix_damage_photo_att exists - skip'';'
);
PREPARE stmt FROM @sql; EXECUTE stmt; DEALLOCATE PREPARE stmt;


-- =====================================================
-- 5) Backfill attachments ownership where possible (existing data)
-- (Safe to run multiple times because it only fills NULL entity_type)
-- =====================================================

-- Dispatch POD
UPDATE attachments a
JOIN distributor_dispatches d ON d.pod_attachment_id = a.id
SET a.entity_type='DISPATCH', a.entity_id=d.id, a.distributor_id=d.distributor_id
WHERE a.entity_type IS NULL;

-- Warehouse return POD
UPDATE attachments a
JOIN warehouse_returns wr ON wr.pod_attachment_id = a.id
SET a.entity_type='WH_RETURN', a.entity_id=wr.id, a.distributor_id=wr.distributor_id
WHERE a.entity_type IS NULL;

-- Outlet delivery POD
UPDATE attachments a
JOIN outlet_deliveries od ON od.pod_attachment_id = a.id
SET a.entity_type='OUTLET_DELIVERY', a.entity_id=od.id, a.distributor_id=od.distributor_id, a.outlet_id=od.outlet_id
WHERE a.entity_type IS NULL;

-- Warehouse receipt POD/image
UPDATE attachments a
JOIN warehouse_receipts w ON w.pod_attachment_id = a.id
SET a.entity_type='WH_RECEIPT', a.entity_id=w.id
WHERE a.entity_type IS NULL;

COMMIT;