<?php
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';

$me = require_role(['DISTRIBUTOR']); // الصفحة للموزعين فقط
$pdo = db();

$distId = (int)($me['distributor_id'] ?? 0);
if ($distId <= 0) {
  http_response_code(400);
  exit("Distributor not linked.");
}

$error = null;
$info  = null;

function fetch_my_outlets(PDO $pdo, int $distId): array {
  $st = $pdo->prepare("
    SELECT o.id, o.name, o.outlet_type, o.address, o.phone
    FROM distributor_outlets do
    JOIN outlets o ON o.id = do.outlet_id
    WHERE do.distributor_id = ?
    ORDER BY o.name
  ");
  $st->execute([$distId]);
  return $st->fetchAll(PDO::FETCH_ASSOC);
}

// معالجة إضافة منفذ جديد
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $op = $_POST['op'] ?? '';

  if ($op === 'create_outlet') {
    $name        = trim($_POST['name'] ?? '');
    $outletType  = $_POST['outlet_type'] ?? 'OTHER';
    $address     = trim($_POST['address'] ?? '');
    $phone       = trim($_POST['phone'] ?? '');

    if ($name === '') {
      $error = "الرجاء إدخال اسم المنفذ.";
    } else {
      if (!in_array($outletType, ['MARKET','BOOKSTORE','SCHOOL','EVENT','OTHER'], true)) {
        $outletType = 'OTHER';
      }

      try {
        $pdo->beginTransaction();

        // إنشاء المنفذ في جدول outlets
        $st = $pdo->prepare("
          INSERT INTO outlets (name, outlet_type, address, phone, default_deal_type, active)
          VALUES (?, ?, ?, ?, 'CONSIGNMENT', 1)
        ");
        $st->execute([$name, $outletType, $address !== '' ? $address : null, $phone !== '' ? $phone : null]);
        $newOutletId = (int)$pdo->lastInsertId();

        // ربط المنفذ بالموزع الحالي
        $st2 = $pdo->prepare("
          INSERT INTO distributor_outlets (distributor_id, outlet_id)
          VALUES (?, ?)
        ");
        $st2->execute([$distId, $newOutletId]);

        $pdo->commit();
        $info = "تم إنشاء المنفذ وربطه بك بنجاح.";
      } catch (Throwable $e) {
        $pdo->rollBack();
        $error = "حدث خطأ أثناء إنشاء المنفذ: " . $e->getMessage();
      }
    }
  }

  if ($op === 'remove_link') {
    $outletId = (int)($_POST['outlet_id'] ?? 0);
    if ($outletId <= 0) {
      $error = "بيانات غير صحيحة.";
    } else {
      try {
        $st = $pdo->prepare("
          DELETE FROM distributor_outlets
          WHERE distributor_id = ? AND outlet_id = ?
        ");
        $st->execute([$distId, $outletId]);
        $info = "تم إزالة هذا المنفذ من قائمتك.";
      } catch (Throwable $e) {
        $error = "خطأ أثناء الحذف: " . $e->getMessage();
      }
    }
  }
}

// تحميل منافذ هذا الموزع
$myOutlets = fetch_my_outlets($pdo, $distId);
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<title>منافذي — MGZ</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<style>
  *, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }
  body {
    font-family: 'Tajawal', sans-serif;
    background: #f3f4f6;
    color: #1f2937;
    min-height: 100vh;
  }
  a { text-decoration: none; }

  .back-link {
    display: inline-block;
    padding: 10px 20px;
    color: #6b7280;
    text-decoration: none;
    font-size: .9rem;
    transition: color .2s;
  }
  .back-link:hover { color: #1d4ed8; }

  .page-header {
    background: linear-gradient(135deg, #1e3a5f 0%, #2563eb 100%);
    color: #fff;
    padding: 22px 32px;
    border-radius: 12px;
    margin: 0 20px 24px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    box-shadow: 0 4px 15px rgba(37,99,235,.35);
  }
  .page-header h1 { font-size: 1.4rem; font-weight: 700; }
  .page-header .subtitle { font-size: .85rem; opacity: .85; margin-top: 4px; }

  .content { padding: 0 20px 40px; max-width: 900px; margin: auto; }

  .card {
    background: #fff;
    border-radius: 10px;
    box-shadow: 0 2px 8px rgba(0,0,0,.08);
    margin-bottom: 24px;
    overflow: hidden;
  }
  .card-header {
    padding: 14px 20px;
    border-bottom: 1px solid #e5e7eb;
    font-size: .95rem;
    font-weight: 600;
    color: #1e3a5f;
    display: flex;
    align-items: center;
    gap: 8px;
  }
  .card-body { padding: 18px 20px 20px; }

  .alert {
    border-radius: 8px;
    padding: 10px 14px;
    font-size: .86rem;
    margin-bottom: 15px;
  }
  .alert-error {
    background: #fef2f2;
    border: 1px solid #fecaca;
    color: #b91c1c;
  }
  .alert-info {
    background: #eff6ff;
    border: 1px solid #bfdbfe;
    color: #1e40af;
  }

  .form-row {
    display: flex;
    flex-wrap: wrap;
    gap: 14px;
    align-items: flex-end;
  }
  .form-row label {
    display: flex;
    flex-direction: column;
    gap: 4px;
    font-size: .82rem;
    color: #374151;
    font-weight: 500;
  }
  .form-row input,
  .form-row select {
    min-width: 220px;
    padding: 8px 12px;
    border-radius: 7px;
    border: 1px solid #d1d5db;
    background: #f9fafb;
    font-size: .9rem;
    outline: none;
  }
  .form-row input:focus,
  .form-row select:focus {
    border-color: #2563eb;
    box-shadow: 0 0 0 3px rgba(37,99,235,.15);
    background: #fff;
  }

  .btn {
    border: none;
    border-radius: 7px;
    padding: 8px 16px;
    font-size: .88rem;
    font-weight: 600;
    cursor: pointer;
    font-family: inherit;
    display: inline-flex;
    align-items: center;
    gap: 6px;
    transition: background .2s, transform .1s;
  }
  .btn:active { transform: scale(.97); }
  .btn-primary   { background: #2563eb; color: #fff; }
  .btn-primary:hover { background: #1d4ed8; }
  .btn-danger    { background: #dc2626; color: #fff; }
  .btn-danger:hover { background: #b91c1c; }

  table { width: 100%; border-collapse: collapse; margin-top: 10px; }
  thead th {
    background: #f1f5f9;
    color: #374151;
    font-weight: 600;
    font-size: .84rem;
    padding: 10px 12px;
    border-bottom: 1px solid #e5e7eb;
    text-align: right;
  }
  tbody td {
    padding: 10px 12px;
    font-size: .86rem;
    border-bottom: 1px solid #f3f4f6;
    color: #374151;
  }
  tbody tr:nth-child(even) { background: #fafafa; }

  .empty-state {
    text-align: center;
    padding: 26px 16px;
    color: #9ca3af;
    font-size: .88rem;
  }
  .empty-state .icon {
    font-size: 2rem;
    margin-bottom: 8px;
  }

  @media (max-width: 700px) {
    .content { padding: 0 12px 28px; }
    .page-header { margin: 0 12px 18px; padding: 16px 18px; flex-direction: column; align-items: flex-start; gap: 8px; }
    .form-row { flex-direction: column; align-items: stretch; }
    .form-row input, .form-row select { width: 100%; min-width: 0; }
  }
</style>

<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
</head>
<body>

<a class="back-link" href="../dashboard.php">← العودة للوحة التحكم</a>

<div class="page-header">
  <div>
    <h1>🏪 منافذي</h1>
    <div class="subtitle">إضافة وإدارة منافذ البيع الخاصة بك كموزع</div>
  </div>
</div>

<div class="content">

  <?php if ($error): ?>
    <div class="alert alert-error"><?= h($error) ?></div>
  <?php elseif ($info): ?>
    <div class="alert alert-info"><?= h($info) ?></div>
  <?php endif; ?>

  <!-- إضافة منفذ جديد -->
  <div class="card">
    <div class="card-header">➕ إضافة منفذ جديد</div>
    <div class="card-body">
      <form method="post" class="form-row">
        <input type="hidden" name="op" value="create_outlet">

        <label>
          اسم المنفذ
          <input type="text" name="name" required placeholder="مثال: كشك أحمد - روي">
        </label>

        <label>
          نوع المنفذ
          <select name="outlet_type">
            <option value="MARKET">سوبر ماركت / بقالة</option>
            <option value="BOOKSTORE">مكتبة</option>
            <option value="SCHOOL">مدرسة</option>
            <option value="EVENT">فعالية / معرض</option>
            <option value="OTHER" selected>آخر</option>
          </select>
        </label>

        <label>
          العنوان (اختياري)
          <input type="text" name="address" placeholder="المنطقة / الشارع / تفاصيل أخرى">
        </label>

        <label>
          رقم الهاتف (اختياري)
          <input type="text" name="phone" placeholder="رقم للتواصل مع المنفذ">
        </label>

        <button type="submit" class="btn btn-primary">
          ✔ حفظ المنفذ
        </button>
      </form>
      <div style="font-size:.8rem;color:#6b7280;margin-top:8px;">
        سيتم إنشاء المنفذ وربطه تلقائياً بحسابك كموزع، ويمكن للإدارة رؤيته ضمن جميع المنافذ.
      </div>
    </div>
  </div>

  <!-- قائمة منافذي -->
  <div class="card">
    <div class="card-header">📋 قائمة منافذي</div>
    <div class="card-body">
      <?php if (empty($myOutlets)): ?>
        <div class="empty-state">
          <div class="icon">🏪</div>
          <div>لا توجد منافذ مرتبطة بك حتى الآن. ابدأ بإضافة منفذ جديد من النموذج أعلاه.</div>
        </div>
      <?php else: ?>
        <table>
          <thead>
            <tr>
              <th>#</th>
              <th>اسم المنفذ</th>
              <th>العنوان</th>
              <th>الهاتف</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($myOutlets as $idx => $o): ?>
              <tr>
                <td><?= $idx + 1 ?></td>
                <td><?= h($o['name']) ?></td>
                <td><?= $o['address'] !== null ? h($o['address']) : '—' ?></td>
                <td><?= $o['phone']   !== null ? h($o['phone'])   : '—' ?></td>
                <td>
                  <form method="post" style="display:inline;" onsubmit="return confirm('إزالة هذا المنفذ من قائمتك؟ لن يتم حذف المنفذ من النظام.');">
                    <input type="hidden" name="op" value="remove_link">
                    <input type="hidden" name="outlet_id" value="<?= (int)$o['id'] ?>">
                    <button type="submit" class="btn btn-danger" style="font-size:.78rem;padding:5px 10px;">
                      🗑 إزالة من منافذي
                    </button>
                  </form>
                </td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      <?php endif; ?>
    </div>
  </div>

</div>
</body>
</html>
