<?php
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/docno.php';

$me  = require_role(['DISTRIBUTOR']);
$pdo = db();

$csrf  = csrf_token();
$tab   = $_GET['tab'] ?? 'issued'; // issued | partial | held
$action = $_GET['action'] ?? 'list';
$errors = [];

$distId = (int)($me['distributor_id'] ?? 0);
if ($distId <= 0) {
    http_response_code(403);
    exit('Distributor account is not linked.');
}

// ─────────────────────────────────────────────────────────────
// Helpers
// ─────────────────────────────────────────────────────────────
function fetch_outlet_name(PDO $pdo, int $outletId): string {
    $st = $pdo->prepare("SELECT name FROM outlets WHERE id=?");
    $st->execute([$outletId]);
    return (string)($st->fetchColumn() ?: '');
}

function fetch_invoice_for_dist(PDO $pdo, int $invoiceId, int $distId): ?array {
    $st = $pdo->prepare(
        "SELECT i.*, o.name AS outlet_name\n"
      . "FROM invoices i\n"
      . "JOIN outlets o ON o.id=i.outlet_id\n"
      . "WHERE i.id=? AND i.distributor_id=?"
    );
    $st->execute([$invoiceId, $distId]);
    $r = $st->fetch();
    return $r ?: null;
}

function fetch_invoices_by_status(PDO $pdo, int $distId, string $status): array {
    $st = $pdo->prepare(
        "SELECT i.id, i.invoice_no, i.issued_at, i.status, i.total_amount, i.total_paid, i.balance_due,\n"
      . "       o.name AS outlet_name\n"
      . "FROM invoices i\n"
      . "JOIN outlets o ON o.id=i.outlet_id\n"
      . "WHERE i.distributor_id=?\n"
      . "  AND i.status=?\n"
      . "  AND i.balance_due > 0\n"
      . "ORDER BY i.issued_at ASC, i.id ASC"
    );
    $st->execute([$distId, $status]);
    return $st->fetchAll();
}

function fetch_held_payments(PDO $pdo, int $distId): array {
    $st = $pdo->prepare(
        "SELECT p.id, p.payment_no, p.paid_at, p.amount, p.method, p.status,\n"
      . "       o.name AS outlet_name,\n"
      . "       COUNT(pa.id) AS alloc_count,\n"
      . "       COALESCE(SUM(pa.amount_allocated),0) AS allocated_sum\n"
      . "FROM payments p\n"
      . "LEFT JOIN outlets o ON o.id=p.outlet_id\n"
      . "LEFT JOIN payment_allocations pa ON pa.payment_id=p.id\n"
      . "WHERE p.distributor_id=? AND p.status='HELD' AND p.method='CASH'\n"
      . "GROUP BY p.id\n"
      . "ORDER BY p.paid_at DESC, p.id DESC\n"
      . "LIMIT 300"
    );
    $st->execute([$distId]);
    return $st->fetchAll();
}

// ─────────────────────────────────────────────────────────────
// POST: collect payment for a single invoice
// ─────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    csrf_check();
    $op = $_POST['op'] ?? '';

    if ($op === 'collect_one') {
        $invoiceId = (int)($_POST['invoice_id'] ?? 0);
        $paid_at   = trim($_POST['paid_at'] ?? '');
        $amount    = (float)($_POST['amount'] ?? 0);
        $notes     = trim($_POST['notes'] ?? '');

        $inv = $invoiceId > 0 ? fetch_invoice_for_dist($pdo, $invoiceId, $distId) : null;
        if (!$inv) {
            $errors[] = 'الفاتورة غير موجودة أو لا تخص حسابك.';
        } elseif (!in_array($inv['status'], ['ISSUED','PARTIAL'], true) || (float)$inv['balance_due'] <= 0) {
            $errors[] = 'هذه الفاتورة ليست ضمن المستحقات الحالية.';
        } elseif ($paid_at === '' || $amount <= 0) {
            $errors[] = 'يرجى إدخال تاريخ التحصيل والمبلغ.';
        } else {
            $bal = round((float)$inv['balance_due'], 3);
            $amt = round($amount, 3);
            if ($amt > $bal) {
                $errors[] = 'المبلغ أكبر من المتبقي على الفاتورة.';
            }
        }

        if (empty($errors)) {
            $pdo->beginTransaction();
            try {
                $payNo = next_doc_no($pdo, 'PAY', $paid_at);

                $pdo->prepare(
                    "INSERT INTO payments\n"
                  . "  (payment_no, outlet_id, distributor_id, amount, method, paid_at, notes, created_by, status)\n"
                  . "VALUES (?, ?, ?, ?, 'CASH', ?, ?, ?, 'HELD')"
                )->execute([
                    $payNo,
                    (int)$inv['outlet_id'],
                    $distId,
                    round($amount, 3),
                    $paid_at,
                    ($notes !== '' ? $notes : 'تحصيل من منفذ (معلّق حتى تسليم المكتب)'),
                    (int)$me['id'],
                ]);

                $paymentId = (int)$pdo->lastInsertId();

                $pdo->prepare(
                    "INSERT INTO payment_allocations (payment_id, invoice_id, amount_allocated)\n"
                  . "VALUES (?, ?, ?)"
                )->execute([$paymentId, (int)$inv['id'], round($amount, 3)]);

                // Update invoice totals (count HELD as paid for outlet side)
                recompute_invoice_totals($pdo, (int)$inv['id'], true);

                $pdo->commit();
                redirect('distributor_receivables.php?tab=' . urlencode($tab));
            } catch (Throwable $e) {
                $pdo->rollBack();
                $errors[] = 'خطأ أثناء الحفظ: ' . $e->getMessage();
            }
        }
    }
}

// ─────────────────────────────────────────────────────────────
// Page data
// ─────────────────────────────────────────────────────────────
$invoiceId = (int)($_GET['invoice_id'] ?? 0);
$invoice   = null;
if ($action === 'collect' && $invoiceId > 0) {
    $invoice = fetch_invoice_for_dist($pdo, $invoiceId, $distId);
    if (!$invoice) {
        $action = 'list';
    }
}

$issued  = [];
$partial = [];
$held    = [];
if ($action === 'list') {
    if ($tab === 'partial') {
        $partial = fetch_invoices_by_status($pdo, $distId, 'PARTIAL');
    } elseif ($tab === 'held') {
        $held = fetch_held_payments($pdo, $distId);
    } else {
        $tab = 'issued';
        $issued = fetch_invoices_by_status($pdo, $distId, 'ISSUED');
    }
}

// Small counters for tabs
$stC = $pdo->prepare("SELECT status, COUNT(*) cnt FROM invoices WHERE distributor_id=? AND status IN ('ISSUED','PARTIAL') AND balance_due>0 GROUP BY status");
$stC->execute([$distId]);
$cnt = ['ISSUED'=>0,'PARTIAL'=>0];
foreach ($stC->fetchAll() as $r) {
    $cnt[$r['status']] = (int)$r['cnt'];
}

$stHeld = $pdo->prepare("SELECT COUNT(*) FROM payments WHERE distributor_id=? AND status='HELD' AND method='CASH'");
$stHeld->execute([$distId]);
$cntHeld = (int)$stHeld->fetchColumn();

?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>مستحقات المنافذ — MGZ</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<style>
*, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }
body{font-family:'Tajawal',sans-serif;background:#f4f7f6;color:#333;direction:rtl;padding:20px;font-size:14px;}
a{color:#1a73e8;text-decoration:none;} a:hover{text-decoration:underline;}
.back-link{margin-bottom:14px;display:inline-block;font-size:.9rem;}
.page-header{background:linear-gradient(135deg,#1a73e8,#0d47a1);color:#fff;padding:18px 24px;border-radius:10px;margin-bottom:20px;display:flex;justify-content:space-between;align-items:center;flex-wrap:wrap;gap:10px;}
.page-header h2{font-size:1.2rem;font-weight:700;margin-bottom:2px;}
.page-header small{opacity:.9;font-size:.85rem;}
.tabs{display:flex;gap:10px;flex-wrap:wrap;margin:14px 0 18px;}
.tab{padding:8px 14px;border-radius:999px;border:1px solid #ddd;background:#fff;font-weight:700;font-size:.85rem;}
.tab.active{background:#e8f0fe;border-color:#1a73e8;color:#0d47a1;}
.badge{background:#eee;padding:2px 10px;border-radius:999px;font-size:.78rem;margin-right:6px;}
.card{background:#fff;border-radius:10px;box-shadow:0 2px 8px rgba(0,0,0,.08);margin-bottom:20px;overflow:hidden;}
.card-header{padding:14px 20px;border-bottom:1px solid #eee;font-weight:700;font-size:.95rem;background:#fafafa;display:flex;justify-content:space-between;align-items:center;}
.card-body{padding:20px;}
.table-wrap{overflow-x:auto;}
table{width:100%;border-collapse:collapse;}
th,td{padding:10px 14px;border-bottom:1px solid #eee;text-align:right;white-space:nowrap;}
th{background:#f8f9fa;font-weight:600;color:#555;font-size:.85rem;}
tbody tr:hover td{background:#fafbff;}
.money{font-weight:800;}
.money-red{color:#ea4335;}
.money-green{color:#34a853;}
.btn{display:inline-flex;align-items:center;gap:6px;padding:7px 14px;border:none;border-radius:6px;cursor:pointer;font-size:.88rem;font-family:inherit;font-weight:700;text-decoration:none;white-space:nowrap;}
.btn-primary{background:#1a73e8;color:#fff;}
.btn-outline{background:transparent;border:1px solid #ccc;color:#555;}
.btn-success{background:#34a853;color:#fff;}
.alert{padding:12px 16px;border-radius:7px;margin-bottom:16px;font-size:.9rem;}
.alert-danger{background:#fce8e6;color:#c62828;border-right:4px solid #ea4335;}
.alert-info{background:#e3f2fd;color:#0d47a1;border-right:4px solid #1a73e8;}
.form-grid{display:grid;grid-template-columns:repeat(auto-fit,minmax(180px,1fr));gap:14px;align-items:end;}
label{display:block;font-weight:700;color:#555;margin-bottom:5px;font-size:.85rem;}
input[type=number],input[type=datetime-local],input[type=text],textarea{width:100%;padding:8px 10px;border:1px solid #ddd;border-radius:6px;font-size:.9rem;font-family:inherit;background:#fff;}
textarea{min-height:70px;}
.empty{padding:36px 10px;text-align:center;color:#999;}
@media (max-width:768px){body{padding:12px;} th,td{padding:8px;font-size:.82rem;}}
</style>

<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
</head>
<body>

<a href="../dashboard.php" class="back-link">&#8594; العودة للرئيسية</a>

<?php foreach ($errors as $e): ?>
  <div class="alert alert-danger">&#9888; <?= h($e) ?></div>
<?php endforeach; ?>

<div class="page-header">
  <div>
    <h2>💰 مستحقات المنافذ</h2>
    <small>متابعة الفواتير غير المسددة والمدفوعة جزئيًا + الدفعات في عهدتك</small>
  </div>
  <a class="btn btn-outline" style="background:rgba(255,255,255,.15);border-color:rgba(255,255,255,.35);color:#fff;" href="distributor_receivables.php">تحديث</a>
</div>

<div class="tabs">
  <a class="tab <?= $tab==='issued'?'active':'' ?>" href="distributor_receivables.php?tab=issued">
    غير مسددة <span class="badge"><?= (int)$cnt['ISSUED'] ?></span>
  </a>
  <a class="tab <?= $tab==='partial'?'active':'' ?>" href="distributor_receivables.php?tab=partial">
    مدفوعة جزئيًا <span class="badge"><?= (int)$cnt['PARTIAL'] ?></span>
  </a>
  <a class="tab <?= $tab==='held'?'active':'' ?>" href="distributor_receivables.php?tab=held">
    دفعات في عهدتي <span class="badge"><?= (int)$cntHeld ?></span>
  </a>
</div>

<?php if ($action === 'collect' && $invoice): ?>

  <div class="card">
    <div class="card-header">تحصيل على فاتورة واحدة</div>
    <div class="card-body">
      <div class="alert alert-info">
        <strong><?= h($invoice['invoice_no']) ?></strong> — <?= h($invoice['outlet_name']) ?>
        <div style="margin-top:6px;">
          الإجمالي: <span class="money"><?= omr((float)$invoice['total_amount']) ?></span> —
          المدفوع: <span class="money money-green"><?= omr((float)$invoice['total_paid']) ?></span> —
          المتبقي: <span class="money money-red"><?= omr((float)$invoice['balance_due']) ?></span>
        </div>
      </div>

      <form method="post">
        <input type="hidden" name="csrf" value="<?= h($csrf) ?>">
        <input type="hidden" name="op" value="collect_one">
        <input type="hidden" name="invoice_id" value="<?= (int)$invoice['id'] ?>">

        <div class="form-grid">
          <div>
            <label>تاريخ ووقت التحصيل *</label>
            <input type="datetime-local" name="paid_at" value="<?= date('Y-m-d\TH:i') ?>" required>
          </div>
          <div>
            <label>المبلغ المقبوض الآن (ر.ع.) *</label>
            <input type="number" name="amount" step="0.001" min="0.001" required
                   value="<?= h((string)round((float)$invoice['balance_due'],3)) ?>">
          </div>
          <div>
            <label>ملاحظات (اختياري)</label>
            <input type="text" name="notes" placeholder="مثال: قبض من المنفذ" value="">
          </div>
        </div>

        <div style="margin-top:16px;display:flex;gap:10px;">
          <button class="btn btn-success" type="submit" onclick="return confirm('تسجيل التحصيل كدفعة معلّقة (HELD) حتى تسليم المكتب؟');">✓ حفظ التحصيل</button>
          <a class="btn btn-outline" href="distributor_receivables.php?tab=<?= h($tab) ?>">إلغاء</a>
        </div>
      </form>
    </div>
  </div>

<?php else: ?>

  <?php if ($tab === 'held'): ?>
    <div class="card">
      <div class="card-header">دفعات كاش في عهدتك (HELD)</div>
      <div class="card-body">
        <?php if (empty($held)): ?>
          <div class="empty">لا توجد دفعات معلّقة حالياً.</div>
        <?php else: ?>
          <div class="table-wrap">
            <table>
              <thead>
                <tr>
                  <th>رقم الدفعة</th>
                  <th>التاريخ</th>
                  <th>المنفذ</th>
                  <th>المبلغ</th>
                  <th>فواتير مرتبطة</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($held as $r): ?>
                  <tr>
                    <td><strong><?= h($r['payment_no']) ?></strong></td>
                    <td><?= h($r['paid_at']) ?></td>
                    <td><?= h($r['outlet_name'] ?? '') ?></td>
                    <td class="money money-green"><?= omr((float)$r['amount']) ?></td>
                    <td style="text-align:center;"><?= (int)$r['alloc_count'] ?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      </div>
    </div>

  <?php else: ?>
    <?php $rows = ($tab === 'partial') ? $partial : $issued; ?>
    <div class="card">
      <div class="card-header"><?= $tab==='partial' ? 'فواتير مدفوعة جزئيًا' : 'فواتير غير مسددة' ?></div>
      <div class="card-body">
        <?php if (empty($rows)): ?>
          <div class="empty">لا توجد فواتير في هذه القائمة.</div>
        <?php else: ?>
          <div class="table-wrap">
            <table>
              <thead>
                <tr>
                  <th>رقم الفاتورة</th>
                  <th>التاريخ</th>
                  <th>المنفذ</th>
                  <th>الإجمالي</th>
                  <th>المدفوع</th>
                  <th>المتبقي</th>
                  <th>إجراء</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($rows as $r): ?>
                  <tr>
                    <td><strong><?= h($r['invoice_no']) ?></strong></td>
                    <td><?= h($r['issued_at']) ?></td>
                    <td><?= h($r['outlet_name']) ?></td>
                    <td><?= omr((float)$r['total_amount']) ?></td>
                    <td class="money money-green"><?= omr((float)$r['total_paid']) ?></td>
                    <td class="money money-red"><?= omr((float)$r['balance_due']) ?></td>
                    <td>
                      <a class="btn btn-primary" href="distributor_receivables.php?action=collect&tab=<?= h($tab) ?>&invoice_id=<?= (int)$r['id'] ?>">💵 تحصيل</a>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      </div>
    </div>
  <?php endif; ?>

<?php endif; ?>

</body>
</html>
