<?php
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/audit.php';

$me  = require_role(['ADMIN','DISTRIBUTOR']);
$pdo = db();
$csrf = csrf_token();

// تم إلغاء ميزة إدارة الأسعار من النظام (الجدول outlet_issue_pricing غير موجود في قاعدة البيانات الحالية)
flash('ميزة إدارة الأسعار تم إلغاؤها من النظام.','warning');
redirect('outlets.php');

$outletId = (int)($_GET['outlet_id'] ?? 0);
if ($outletId <= 0) {
  http_response_code(400);
  exit('outlet_id مطلوب');
}

// distributor can only access his outlets
if ($me['role'] === 'DISTRIBUTOR') {
  if (empty($me['distributor_id'])) exit('الموزع غير مرتبط بحساب موزع');
  $st = $pdo->prepare("SELECT COUNT(*) FROM distributor_outlets WHERE distributor_id=? AND outlet_id=?");
  $st->execute([(int)$me['distributor_id'], $outletId]);
  if ((int)$st->fetchColumn() <= 0) exit('هذا المنفذ غير مرتبط بالموزع');
}

function outlet_name(PDO $pdo, int $outletId): string {
  $st = $pdo->prepare("SELECT name FROM outlets WHERE id=?");
  $st->execute([$outletId]);
  return (string)($st->fetchColumn() ?: '');
}

// Save
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $issueIds  = $_POST['issue_id'] ?? [];
  $modes     = $_POST['price_mode'] ?? [];
  $values    = $_POST['value'] ?? [];

  if (!is_array($issueIds) || !is_array($modes) || !is_array($values)) {
    flash('بيانات غير صحيحة','danger');
    redirect('outlet_pricing.php?outlet_id='.$outletId);
  }

  $pdo->beginTransaction();
  try {
    for ($k=0; $k<count($issueIds); $k++) {
      $iid = (int)$issueIds[$k];
      if ($iid<=0) continue;
      $mode = strtoupper(trim((string)($modes[$k] ?? 'COVER')));
      if (!in_array($mode, ['COVER','SUPPLY_PRICE','DISCOUNT_PERCENT'], true)) $mode = 'COVER';
      $valRaw = trim((string)($values[$k] ?? ''));
      $val = ($valRaw === '') ? null : (float)$valRaw;

      // Normalize values
      if ($mode === 'COVER') {
        $val = null;
      } elseif ($mode === 'DISCOUNT_PERCENT') {
        $val = max(0.0, min(100.0, (float)($val ?? 0)));
      } else { // SUPPLY_PRICE
        $val = max(0.0, (float)($val ?? 0));
      }

      // one row per day (upsert)
      $sql = "INSERT INTO outlet_issue_pricing (outlet_id, issue_id, price_mode, value, valid_from, valid_to)
              VALUES (?,?,?,?, CURDATE(), NULL)
              ON DUPLICATE KEY UPDATE price_mode=VALUES(price_mode), value=VALUES(value), valid_to=NULL";
      $pdo->prepare($sql)->execute([$outletId, $iid, $mode, $val]);
    }

    audit_log($pdo, (int)$me['id'], 'UPDATE', 'outlet_issue_pricing', $outletId, ['outlet_id'=>$outletId]);
    $pdo->commit();
    flash('تم حفظ التسعير بنجاح','success');
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash($e->getMessage(),'danger');
  }
  redirect('outlet_pricing.php?outlet_id='.$outletId);
}

// Build grid: issues + current pricing (latest valid row as of today)
$issues = $pdo->query("SELECT i.id, i.product_id, p.name AS product_name, i.issue_number, i.issue_month, i.issue_year, i.cover_price
                        FROM issues i JOIN products p ON p.id=i.product_id
                        ORDER BY p.name, i.issue_year DESC, i.issue_month DESC, i.issue_number DESC")->fetchAll();

$st = $pdo->prepare("SELECT oip.*
                     FROM outlet_issue_pricing oip
                     WHERE oip.outlet_id=?
                       AND (oip.valid_from IS NULL OR oip.valid_from <= CURDATE())
                       AND (oip.valid_to IS NULL OR oip.valid_to >= CURDATE())
                     ORDER BY oip.issue_id, COALESCE(oip.valid_from,'1970-01-01') DESC, oip.id DESC");
$st->execute([$outletId]);
$pricingRows = $st->fetchAll();

$pricing = [];
foreach ($pricingRows as $r) {
  $iid = (int)$r['issue_id'];
  if (!isset($pricing[$iid])) $pricing[$iid] = $r; // latest
}

$outletName = outlet_name($pdo, $outletId);

?><!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>تسعير المنافذ — MGZ</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
  <style>
    body{font-family:Tajawal,Tahoma,Arial; background:#f4f7f6; margin:0; padding:16px;}
    .topbar{display:flex; align-items:center; justify-content:space-between; gap:10px; flex-wrap:wrap; margin-bottom:12px;}
    .btn{background:#1a73e8; color:#fff; border:none; padding:10px 14px; border-radius:10px; cursor:pointer; font-size:14px;}
    .btn-secondary{background:#6c757d;}
    .card{background:#fff; border-radius:14px; padding:14px; box-shadow:0 2px 10px rgba(0,0,0,.06);}
    table{width:100%; border-collapse:collapse;}
    th,td{padding:10px; border-bottom:1px solid #eee; font-size:13px; text-align:right; vertical-align:middle;}
    th{background:#fafafa; position:sticky; top:0; z-index:1;}
    select,input{width:100%; padding:8px 10px; border:1px solid #ddd; border-radius:10px; font-size:13px;}
    .muted{color:#666; font-size:12px;}
    .row{display:flex; gap:10px; align-items:center; flex-wrap:wrap;}
  </style>

<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
</head>
<body>

<div class="topbar">
  <a class="btn btn-secondary" style="text-decoration:none;" href="outlets.php">⬅ العودة للمنافذ</a>
  <div style="flex:1"></div>
  <div class="muted">المنفذ: <b><?=h($outletName)?></b></div>
</div>

<?php show_flash(); ?>

<div class="card">
  <div class="row" style="margin-bottom:10px;">
    <div style="font-weight:700;">💰 تسعير الإصدارات للمنفذ</div>
    <div class="muted">يُستخدم التسعير في تسليم المنافذ (ODL) والتسويات والفواتير.</div>
  </div>

  <form method="post">
    <input type="hidden" name="csrf" value="<?=h($csrf)?>">

    <table>
      <thead>
        <tr>
          <th style="width:28%">المنتج</th>
          <th style="width:28%">الإصدار</th>
          <th style="width:12%">سعر الغلاف</th>
          <th style="width:18%">نمط التسعير</th>
          <th style="width:14%">القيمة</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($issues as $i):
        $iid = (int)$i['id'];
        $label = 'عدد '.($i['issue_number'] ?? '').' - '.str_pad((string)$i['issue_month'],2,'0',STR_PAD_LEFT).'/'.$i['issue_year'];
        $cur = $pricing[$iid] ?? null;
        $mode = $cur['price_mode'] ?? 'COVER';
        $val  = $cur['value'] ?? '';
      ?>
        <tr>
          <td><?=h($i['product_name'])?></td>
          <td><?=h($label)?></td>
          <td><?=h(fmt_num($i['cover_price']))?> ر.ع</td>
          <td>
            <input type="hidden" name="issue_id[]" value="<?= $iid ?>">
            <select name="price_mode[]">
              <option value="COVER" <?= $mode==='COVER'?'selected':'' ?>>سعر الغلاف</option>
              <option value="SUPPLY_PRICE" <?= $mode==='SUPPLY_PRICE'?'selected':'' ?>>سعر توريد ثابت</option>
              <option value="DISCOUNT_PERCENT" <?= $mode==='DISCOUNT_PERCENT'?'selected':'' ?>>خصم % من الغلاف</option>
            </select>
          </td>
          <td>
            <input name="value[]" value="<?=h($val)?>" placeholder="مثال: 0.5 أو 10">
            <div class="muted">اتركها فارغة عند “سعر الغلاف”.</div>
          </td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>

    <div style="margin-top:12px; display:flex; gap:10px; flex-wrap:wrap;">
      <button class="btn" type="submit">💾 حفظ التسعير</button>
      <a class="btn btn-secondary" style="text-decoration:none;" href="outlet_deliveries.php">الانتقال لتسليم المنافذ (ODL)</a>
    </div>
  </form>
</div>

</body>
</html>
