<?php
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/csrf.php';

// شاشة الإدارة: هدفها "ترحيل توريدات الكاش" التي قام الموزع بتحصيلها من المنافذ.
// لا تسمح بإنشاء تحصيل جديد من المكتب على المنافذ (لتفادي التكرار).

$me  = require_role(['ADMIN','WAREHOUSE']);
$pdo = db();

$csrf   = csrf_token();
$action = $_GET['action'] ?? 'list';
$errors = [];

$METHOD_AR = [
    'CASH'     => 'نقداً',
    'TRANSFER' => 'تحويل بنكي',
    'OTHER'    => 'أخرى',
];

/* ══════════════════════════════════════════
   دوال مساعدة
══════════════════════════════════════════ */
function fetch_distributors(PDO $pdo): array {
    return $pdo->query("SELECT id, name FROM distributors WHERE active=1 ORDER BY name")
        ->fetchAll();
}

// دفعات كاش معلّقة (في عهدة الموزع) جاهزة للترحيل للمكتب

function fetch_held_cash_grouped(PDO $pdo, int $distId): array {
    $st = $pdo->prepare("
        SELECT
            DATE(p.paid_at) AS paid_day,
            o.id            AS outlet_id,
            o.name          AS outlet_name,
            p.id            AS payment_id,
            p.payment_no,
            p.paid_at,
            p.amount        AS payment_amount,
            GROUP_CONCAT(DISTINCT i.invoice_no ORDER BY i.invoice_no SEPARATOR ', ') AS invoices_list,
            SUM(CASE WHEN i.status IN ('PAID','PARTIAL') THEN 0 ELSE 1 END) AS bad_invoices
        FROM payments p
        LEFT JOIN outlets o ON o.id = p.outlet_id
        LEFT JOIN payment_allocations pa ON pa.payment_id = p.id
        LEFT JOIN invoices i ON i.id = pa.invoice_id
        WHERE p.distributor_id = ?
          AND p.status = 'HELD'
          AND p.method = 'CASH'
        GROUP BY DATE(p.paid_at), o.id, p.id
        HAVING COUNT(i.id) > 0 AND bad_invoices = 0
        ORDER BY DATE(p.paid_at) ASC, o.name ASC, p.paid_at ASC, p.id ASC
    ");
    $st->execute([$distId]);
    $rows = $st->fetchAll();

    // Build nested structure: day -> outlet -> payments
    $out = [
        'totals' => ['count'=>0,'amount'=>0.0],
        'days' => []
    ];

    foreach ($rows as $r) {
        $day = $r['paid_day'] ?: '—';
        $oid = (int)($r['outlet_id'] ?? 0);
        $onm = $r['outlet_name'] ?? '—';

        if (!isset($out['days'][$day])) {
            $out['days'][$day] = [
                'totals' => ['count'=>0,'amount'=>0.0],
                'outlets' => []
            ];
        }
        if (!isset($out['days'][$day]['outlets'][$oid])) {
            $out['days'][$day]['outlets'][$oid] = [
                'outlet_name' => $onm,
                'totals' => ['count'=>0,'amount'=>0.0],
                'payments' => []
            ];
        }

        $amt = (float)$r['payment_amount'];
        $out['totals']['count']++;
        $out['totals']['amount'] += $amt;

        $out['days'][$day]['totals']['count']++;
        $out['days'][$day]['totals']['amount'] += $amt;

        $out['days'][$day]['outlets'][$oid]['totals']['count']++;
        $out['days'][$day]['outlets'][$oid]['totals']['amount'] += $amt;

        $out['days'][$day]['outlets'][$oid]['payments'][] = [
            'payment_id' => (int)$r['payment_id'],
            'payment_no' => $r['payment_no'],
            'paid_at' => $r['paid_at'],
            'amount' => $amt,
            'invoices_list' => $r['invoices_list'] ?? ''
        ];
    }

    return $out;
}

/* ══════════════════════════════════════════
   معالجة POST
══════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    csrf_check();
    if (!consume_form_token("payment")) { die("تم إرسال الطلب مسبقاً."); }
$op = $_POST['op'] ?? '';

    // ترحيل مجموعة دفعات لموزع (توريد للمكتب)
    if ($op === 'post_selected') {
        $dist_id = (int)($_POST['distributor_id'] ?? 0);
        $ids = $_POST['payment_ids'] ?? [];
        $ids = array_values(array_filter(array_map('intval', (array)$ids)));

        if ($dist_id <= 0) {
            $errors[] = 'يرجى اختيار الموزع.';
        } elseif (empty($ids)) {
            $errors[] = 'يرجى اختيار دفعة واحدة على الأقل للترحيل.';
        } else {
            $pdo->beginTransaction();
            try {
                $in = implode(',', array_fill(0, count($ids), '?'));

                // حدّث الحالة إلى POSTED
                $params = array_merge([(int)$me['id']], $ids, [$dist_id]);
                $pdo->prepare("
                    UPDATE payments
                    SET status='POSTED', posted_by=?, posted_at=NOW()
                    WHERE id IN ($in)
                      AND status='HELD'
                      AND distributor_id=?
                ")->execute($params);

                // إعادة احتساب الفواتير المرتبطة (لضمان صحة الحالة/الرصيد)
                $stInv = $pdo->prepare("
                    SELECT DISTINCT pa.invoice_id
                    FROM payment_allocations pa
                    WHERE pa.payment_id IN ($in)
                ");
                $stInv->execute($ids);
                $invIds = $stInv->fetchAll(PDO::FETCH_COLUMN);
                foreach ($invIds as $invId) {
                    // في منطق الشركة: احتساب HELD كمدفوع لأن المنفذ دفع للموزع بالفعل.
                    recompute_invoice_totals($pdo, (int)$invId, true);
                }

                $pdo->commit();
                redirect('payments.php');
            } catch (Throwable $e) {
                $pdo->rollBack();
                $errors[] = 'خطأ أثناء الترحيل: ' . $e->getMessage();
            }
        }
    }
}

/* ══════════════════════════════════════════
   تحضير البيانات
══════════════════════════════════════════ */
if ($action === 'new') {
    $dist_id = (int)($_GET['distributor_id'] ?? 0);
    $distributors = fetch_distributors($pdo);
    $heldG = $dist_id > 0 ? fetch_held_cash_grouped($pdo, $dist_id) : null;
    $rows = [];
} else {
    $dist_id = 0;
    $distributors = [];
    $heldG = null;

    // قائمة التحصيلات: تعرض فقط ما تم ترحيله للمكتب (POSTED)
    $rows = $pdo->query("
        SELECT p.id, p.payment_no, p.paid_at, p.method, p.amount, p.status,
               o.name AS outlet_name,
               d.name AS distributor_name,
               u.name AS created_by_name,
               COUNT(pa.id) AS alloc_count
        FROM payments p
        LEFT JOIN outlets o ON o.id = p.outlet_id
        LEFT JOIN distributors d ON d.id = p.distributor_id
        JOIN users u ON u.id = p.created_by
        LEFT JOIN payment_allocations pa ON pa.payment_id = p.id
        WHERE p.status='POSTED'
        GROUP BY p.id
        ORDER BY p.id DESC
        LIMIT 200
    ")->fetchAll();
}

?><!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>توريدات الموزعين (PAY) — MGZ</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<style>
*, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }
body {
    font-family: 'Tajawal', sans-serif;
    background: #f4f7f6;
    color: #333;
    direction: rtl;
    padding: 20px;
    font-size: 14px;
}
a { color: #1a73e8; text-decoration: none; }
a:hover { text-decoration: underline; }
.back-link { margin-bottom: 14px; display: inline-block; font-size: .9rem; }
.page-header {
    background: linear-gradient(135deg, #34a853, #0d652d);
    color: #fff;
    padding: 18px 24px;
    border-radius: 10px;
    margin-bottom: 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 10px;
}
.page-header h2 { font-size: 1.3rem; font-weight: 700; margin-bottom: 2px; }
.page-header small { opacity: .85; font-size: .85rem; }
.card {
    background: #fff;
    border-radius: 10px;
    box-shadow: 0 2px 8px rgba(0,0,0,.08);
    margin-bottom: 20px;
    overflow: hidden;
}
.card-header {
    padding: 14px 20px;
    border-bottom: 1px solid #eee;
    font-weight: 700;
    font-size: .95rem;
    background: #fafafa;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.card-body { padding: 20px; }
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px,1fr));
    gap: 14px;
    margin-bottom: 20px;
}
.stat-card {
    background: #fff;
    border-radius: 10px;
    box-shadow: 0 2px 8px rgba(0,0,0,.08);
    text-align: center;
    padding: 16px 10px;
}
.stat-card .num { font-size: 1.5rem; font-weight: 700; }
.stat-card .lbl { color: #888; font-size: .82rem; margin-top: 3px; }
.table-wrap { overflow-x: auto; }
table { width: 100%; border-collapse: collapse; }
th, td { padding: 10px 14px; border-bottom: 1px solid #eee; text-align: right; white-space: nowrap; }
th { background: #f8f9fa; font-weight: 600; color: #555; font-size: .85rem; }
tr:last-child td { border-bottom: none; }
tbody tr:hover td { background: #fafbff; }
.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(180px,1fr));
    gap: 14px;
    align-items: end;
    margin-bottom: 20px;
}
.form-group label {
    display: block;
    font-weight: 600;
    color: #555;
    margin-bottom: 5px;
    font-size: .85rem;
}
input[type=text], input[type=number], input[type=datetime-local],
input[type=file], select, textarea {
    width: 100%;
    padding: 8px 10px;
    border: 1px solid #ddd;
    border-radius: 6px;
    font-size: .9rem;
    font-family: inherit;
    background: #fff;
    transition: border-color .2s;
}
input:focus, select:focus {
    outline: none;
    border-color: #34a853;
    box-shadow: 0 0 0 3px rgba(52,168,83,.1);
}
.btn {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    padding: 7px 16px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: .88rem;
    font-family: inherit;
    font-weight: 600;
    text-decoration: none;
    transition: opacity .15s;
    white-space: nowrap;
}
.btn:hover { opacity: .88; }
.btn-primary { background: #1a73e8; color: #fff; }
.btn-success { background: #34a853; color: #fff; }
.btn-outline { background: transparent; border: 1px solid #ccc; color: #555; }
.btn-sm      { padding: 4px 10px; font-size: .8rem; }
.alert {
    padding: 12px 16px;
    border-radius: 7px;
    margin-bottom: 16px;
    font-size: .9rem;
}
.alert-danger  { background: #fce8e6; color: #c62828; border-right: 4px solid #ea4335; }
.alert-info    { background: #e3f2fd; color: #0d47a1; border-right: 4px solid #1a73e8; }
.alert-warning { background: #fff8e1; color: #e65100; border-right: 4px solid #fbbc04; }
.money       { font-weight: 700; color: #1a73e8; }
.money-green { font-weight: 700; color: #34a853; }
.money-red   { font-weight: 700; color: #ea4335; }
.empty-state { text-align: center; padding: 50px 20px; color: #aaa; }
.empty-state .icon { font-size: 3rem; margin-bottom: 10px; }
.total-row td { font-weight: 700; background: #f0fff4; }
@media (max-width: 768px) {
    body { padding: 12px; }
    .page-header { flex-direction: column; align-items: flex-start; }
    .form-grid { grid-template-columns: 1fr; }
    .stats-grid { grid-template-columns: repeat(2,1fr); }
    th, td { padding: 8px; font-size: .82rem; }
}
</style>

<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
</head>
<body>

<a href="../dashboard.php" class="back-link">&#8594; العودة للرئيسية</a>

<?php foreach ($errors as $e): ?>
    <div class="alert alert-danger">&#9888; <?= h($e) ?></div>
<?php endforeach; ?>

<?php if ($action === 'new'): ?>

<div class="page-header">
    <div>
        <h2>&#128178; ترحيل توريد موزع</h2>
        <small>اختر الموزع ثم رحّل دفعات الكاش المعلقة (في عهدة الموزع) إلى المكتب</small>
    </div>
    <a href="payments.php" class="btn btn-outline" style="background:rgba(255,255,255,.2);color:#fff;border-color:rgba(255,255,255,.4);">
        &#8594; قائمة التوريدات المُرحّلة
    </a>
</div>

<div class="card">
    <div class="card-header">&#128101; اختيار الموزع</div>
    <div class="card-body">
        <form method="get">
            <input type="hidden" name="action" value="new">
            <div class="form-grid" style="grid-template-columns:1fr auto;">
                <div class="form-group">
                    <label>الموزع *</label>
                    <select name="distributor_id" required>
                        <option value="0">— اختر الموزع —</option>
                        <?php foreach ($distributors as $d): ?>
                        <option value="<?= (int)$d['id'] ?>" <?= $dist_id === (int)$d['id'] ? 'selected' : '' ?>>
                            <?= h($d['name']) ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>&nbsp;</label>
                    <button type="submit" class="btn btn-primary" style="width:100%;">&#128269; تحميل الدفعات</button>
                </div>
            </div>
        </form>
    </div>
</div>

<?php if ($dist_id > 0): ?>
<div class="card">
    <div class="card-header">&#128181; الدفعات المعلقة (كاش)</div>
    <div class="card-body">

        <?php if (!$heldG || empty($heldG['days'])): ?>
            <div class="alert alert-warning">&#9888; لا توجد دفعات كاش معلّقة لهذا الموزع.</div>
        <?php else: ?>

        <div class="alert alert-info">
            إجمالي الدفعات المعلّقة: <strong><?= (int)$heldG['totals']['count'] ?></strong>
            — الإجمالي: <strong><?= omr((float)$heldG['totals']['amount']) ?></strong>
        </div>

        <form method="post">
            <?= csrf_field() ?>
            <input type="hidden" name="form_token" value="<?= h(form_token("payment")) ?>">
            <input type="hidden" name="op" value="post_selected">
            <input type="hidden" name="distributor_id" value="<?= (int)$dist_id ?>">

            <?php foreach ($heldG['days'] as $day => $dayBlock): ?>
                <details open style="margin:10px 0;background:#fff;border:1px solid #eee;border-radius:10px;padding:10px;">
                    <summary style="cursor:pointer;font-weight:800;">
                        &#128197; <?= h($day) ?>
                        — عدد: <?= (int)$dayBlock['totals']['count'] ?>
                        — إجمالي: <?= omr((float)$dayBlock['totals']['amount']) ?>
                    </summary>

                    <?php foreach ($dayBlock['outlets'] as $oid => $oBlock): ?>
                        <details open style="margin:10px 0;background:#fafafa;border:1px solid #eee;border-radius:10px;padding:10px;">
                            <summary style="cursor:pointer;font-weight:700;">
                                &#127978; <?= h($oBlock['outlet_name']) ?>
                                — عدد: <?= (int)$oBlock['totals']['count'] ?>
                                — إجمالي: <?= omr((float)$oBlock['totals']['amount']) ?>
                            </summary>

                            <div class="table-wrap" style="margin-top:10px;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th style="width:40px; text-align:center;">&#10003;</th>
                                            <th>رقم الدفعة</th>
                                            <th>الوقت</th>
                                            <th>الفواتير</th>
                                            <th>المبلغ</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($oBlock['payments'] as $p): ?>
                                        <tr>
                                            <td style="text-align:center;">
                                                <input type="checkbox" name="payment_ids[]" value="<?= (int)$p['payment_id'] ?>" style="width:18px;height:18px;cursor:pointer;">
                                            </td>
                                            <td><strong><?= h($p['payment_no']) ?></strong></td>
                                            <td><?= h($p['paid_at']) ?></td>
                                            <td style="max-width:420px;white-space:normal;"><?= h($p['invoices_list']) ?></td>
                                            <td class="money-green"><?= omr((float)$p['amount']) ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </details>
                    <?php endforeach; ?>

                </details>
            <?php endforeach; ?>

            <div style="margin-top:20px;display:flex;gap:10px;align-items:center;">
                <button type="submit" class="btn btn-success" onclick="return confirm('ترحيل الدفعات المختارة كإيداع تم تسليمه للمكتب؟');">
                    &#10003; ترحيل الدفعات المختارة
                </button>
                <a href="payments.php" class="btn btn-outline">إلغاء</a>
            </div>
        </form>
        <?php endif; ?>

    </div>
</div>
<?php endif; ?>

<?php else: ?>

<div class="page-header">
    <div>
        <h2>&#128178; التوريدات المُرحّلة (PAY)</h2>
        <small>قائمة الدفعات التي تم استلامها وترحيلها للمكتب</small>
    </div>
    <a href="payments.php?action=new" class="btn btn-outline" style="background:rgba(255,255,255,.2);color:#fff;border-color:rgba(255,255,255,.4);">
        &#43; ترحيل توريد موزع
    </a>
</div>

<?php
$stats = $pdo->query("
    SELECT
        COUNT(*) AS total,
        COALESCE(SUM(amount),0) AS sum_amount,
        SUM(method='CASH') AS cash_count,
        SUM(method='TRANSFER') AS transfer_count
    FROM payments
    WHERE status='POSTED'
")->fetch();
?>
<div class="stats-grid">
    <div class="stat-card">
        <div class="num" style="color:#555;"><?= (int)$stats['total'] ?></div>
        <div class="lbl">إجمالي التوريدات المُرحّلة</div>
    </div>
    <div class="stat-card">
        <div class="num" style="color:#34a853;font-size:1.2rem;">
            <?= omr((float)$stats['sum_amount']) ?>
        </div>
        <div class="lbl">إجمالي المُرحّل (ر.ع.)</div>
    </div>
    <div class="stat-card">
        <div class="num" style="color:#1a73e8;"><?= (int)$stats['cash_count'] ?></div>
        <div class="lbl">نقداً</div>
    </div>
    <div class="stat-card">
        <div class="num" style="color:#6f42c1;"><?= (int)$stats['transfer_count'] ?></div>
        <div class="lbl">تحويل بنكي</div>
    </div>
</div>

<div class="card">
    <div class="card-header">
        قائمة التوريدات المُرحّلة
        <span style="font-weight:400;font-size:.85rem;color:#888;"><?= count($rows) ?> توريد</span>
    </div>
    <div class="table-wrap">
        <table>
            <thead>
                <tr>
                    <th>#</th>
                    <th>رقم التوريد</th>
                    <th>التاريخ</th>
                    <th>الموزع</th>
                    <th>المنفذ</th>
                    <th>طريقة الدفع</th>
                    <th>المبلغ</th>
                    <th>الفواتير المرتبطة</th>
                    <th>أنشئ بواسطة</th>
                </tr>
            </thead>
            <tbody>
            <?php if (empty($rows)): ?>
                <tr><td colspan="9">
                    <div class="empty-state">
                        <div class="icon">&#128178;</div>
                        <div>لا توجد توريدات مُرحّلة بعد</div>
                    </div>
                </td></tr>
            <?php else: ?>
                <?php foreach ($rows as $idx => $r): ?>
                <tr>
                    <td><?= $idx + 1 ?></td>
                    <td><strong><?= h($r['payment_no']) ?></strong></td>
                    <td><?= h($r['paid_at']) ?></td>
                    <td><?= h($r['distributor_name'] ?? '') ?></td>
                    <td><?= h($r['outlet_name'] ?? '') ?></td>
                    <td>
                        <?php
                        $methodIcons = ['CASH'=>'&#128181;','TRANSFER'=>'&#127974;','OTHER'=>'&#128260;'];
                        echo ($methodIcons[$r['method']] ?? '') . ' ' . ($METHOD_AR[$r['method']] ?? h($r['method']));
                        ?>
                    </td>
                    <td class="money-green"><?= omr((float)$r['amount']) ?></td>
                    <td style="text-align:center;">
                        <span style="background:#e8f0fe;color:#1a73e8;padding:2px 10px;border-radius:12px;font-size:.82rem;font-weight:700;">
                            <?= (int)$r['alloc_count'] ?> فاتورة
                        </span>
                    </td>
                    <td><?= h($r['created_by_name']) ?></td>
                </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php endif; ?>

</body>
</html>