<?php
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';

require_role(['ADMIN']);

$pdo = db();

function _parse_date_input($s) {
  $s = trim((string)$s);
  if ($s === '') return null;
  // Accept common browser/admin inputs.
  foreach (['Y-m-d', 'm/d/Y', 'd/m/Y'] as $fmt) {
    $d = DateTime::createFromFormat($fmt, $s);
    if ($d && $d->format($fmt) === $s) return $d;
  }
  return null;
}

$from_raw = $_GET['from'] ?? '';
$to_raw   = $_GET['to'] ?? '';

$from_d = _parse_date_input($from_raw);
$to_d   = _parse_date_input($to_raw);

// Unified inclusive range: start-of-day .. end-of-day
$from_dt = $from_d ? ($from_d->format('Y-m-d') . ' 00:00:00') : null;
$to_dt   = $to_d   ? ($to_d->format('Y-m-d')   . ' 23:59:59') : null;

// Values for <input type="date"> must be Y-m-d
$from_date = $from_d ? $from_d->format('Y-m-d') : '';
$to_date   = $to_d   ? $to_d->format('Y-m-d')   : '';

function fmt3($n) { return number_format((float)$n, 3, '.', ''); }
function fmt_omr($n) {
  $n = (float)$n;
  // trim trailing zeros
  $s = number_format($n, 3, '.', '');
  $s = rtrim(rtrim($s, '0'), '.');
  return $s === '' ? '0' : $s;
}

$issue_id = (int)($_GET['issue_id'] ?? 0);

function issue_label($it) {
  $product = trim((string)($it['product_name'] ?? ''));
  $title = trim((string)($it['title'] ?? ''));
  $base = $product !== '' ? $product : ($title !== '' ? $title : 'إصدار');
  $m = (int)($it['issue_month'] ?? 0);
  $y = (int)($it['issue_year'] ?? 0);
  $datePart = ($m > 0 && $y > 0) ? (' - ' . str_pad((string)$m, 2, '0', STR_PAD_LEFT) . '/' . $y) : '';
  $num = $it['issue_number'] ?? null;
  $numPart = ($num !== null && $num !== '') ? (' - عدد ' . (int)$num) : '';
  return $base . $datePart . $numPart;
}

$issues = $pdo->query("SELECT i.id, p.name AS product_name, i.title, i.issue_month, i.issue_year, i.issue_number, i.cover_price FROM issues i JOIN products p ON p.id=i.product_id ORDER BY i.issue_year DESC, i.issue_month DESC, i.id DESC")->fetchAll(PDO::FETCH_ASSOC);
$issue = null;
foreach ($issues as $it) { if ((int)$it['id'] === $issue_id) { $issue = $it; break; } }

$now = (new DateTime('now'))->format('d/m/Y H:i');

$metrics = [
  'whr_in' => 0,
  'dis_out' => 0,
  'wrt_in' => 0,
  'sold_direct_qty' => 0,
  'sold_direct_val' => 0.0,
  'sold_cons_qty' => 0,
  'sold_cons_val' => 0.0,
  'dam_pending' => 0,
  'dam_approved' => 0,
  'dam_rejected' => 0,
  'inv_count' => 0,
  'inv_total' => 0.0,
  'paid_outlets' => 0.0,
  'paid_office' => 0.0,
  'inv_due' => 0.0,
];

if ($issue_id > 0) {
  // Ledger quantities (warehouse)
  $params = [$issue_id];
  $sql = "SELECT
    COALESCE(SUM(CASE WHEN entity_type='WAREHOUSE' AND ref_type='WH_RECEIPT' THEN qty_in ELSE 0 END),0) AS whr_in,
    COALESCE(SUM(CASE WHEN entity_type='WAREHOUSE' AND ref_type='DIST_ORDER' THEN qty_out ELSE 0 END),0) AS dis_out,
    COALESCE(SUM(CASE WHEN entity_type='WAREHOUSE' AND ref_type='WRT' THEN qty_in ELSE 0 END),0) AS wrt_in
  FROM inventory_ledger
  WHERE status='POSTED' AND issue_id=?";
  if ($from_dt) { $sql .= " AND created_at >= ?"; $params[] = $from_dt; }
  if ($to_dt)   { $sql .= " AND created_at <= ?"; $params[] = $to_dt; }
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  $metrics['whr_in'] = (int)$row['whr_in'];
  $metrics['dis_out'] = (int)$row['dis_out'];
  $metrics['wrt_in'] = (int)$row['wrt_in'];

  // Direct sales
  $params = [$issue_id];
  // outlet_delivery_items uses qty_delivered (not qty)
  $sql = "SELECT
    COALESCE(SUM(odi.qty_delivered),0) AS qty,
    COALESCE(SUM(odi.qty_delivered * odi.unit_price_applied),0) AS val
  FROM outlet_delivery_items odi
  JOIN outlet_deliveries od ON od.id = odi.delivery_id
  WHERE odi.issue_id=? AND od.deal_type='DIRECT_SALE'";
  if ($from_dt) { $sql .= " AND od.delivered_at >= ?"; $params[] = $from_dt; }
  if ($to_dt)   { $sql .= " AND od.delivered_at <= ?"; $params[] = $to_dt; }
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  $metrics['sold_direct_qty'] = (int)$row['qty'];
  $metrics['sold_direct_val'] = (float)$row['val'];

  // Consignment sold (from settlements): include SUBMITTED + APPROVED to show actual settlement entries
  $params = [$issue_id];
  $sql = "SELECT
    COALESCE(SUM(l.qty_sold),0) AS qty,
    COALESCE(SUM(l.qty_sold * l.unit_price_applied),0) AS val
  FROM outlet_settlement_lines l
  JOIN outlet_settlements s ON s.id = l.settlement_id
  WHERE l.issue_id=? AND s.status IN ('SUBMITTED','APPROVED')";
  if ($from_dt) { $sql .= " AND s.settlement_at >= ?"; $params[] = $from_dt; }
  if ($to_dt)   { $sql .= " AND s.settlement_at <= ?"; $params[] = $to_dt; }
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  $metrics['sold_cons_qty'] = (int)$row['qty'];
  $metrics['sold_cons_val'] = (float)$row['val'];

  // Damages by status
  $params = [$issue_id];
  $sql = "SELECT
    COALESCE(SUM(CASE WHEN status='PENDING' THEN qty ELSE 0 END),0) AS pending_qty,
    COALESCE(SUM(CASE WHEN status='APPROVED' THEN qty ELSE 0 END),0) AS approved_qty,
    COALESCE(SUM(CASE WHEN status='REJECTED' THEN qty ELSE 0 END),0) AS rejected_qty
  FROM damage_reports
  WHERE issue_id=?";
  if ($from_dt) { $sql .= " AND created_at >= ?"; $params[] = $from_dt; }
  if ($to_dt)   { $sql .= " AND created_at <= ?"; $params[] = $to_dt; }
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  $metrics['dam_pending'] = (int)$row['pending_qty'];
  $metrics['dam_approved'] = (int)$row['approved_qty'];
  $metrics['dam_rejected'] = (int)$row['rejected_qty'];

  // Invoices for this issue (direct + settlement that include this issue)
  $params = [$issue_id, $issue_id];
  $sql = "SELECT DISTINCT i.id, i.invoice_no, i.outlet_id, i.distributor_id, i.issued_at, i.total_amount
          FROM invoices i
          LEFT JOIN outlet_delivery_items odi ON (i.source_type='DIRECT_SALE' AND odi.delivery_id=i.source_id AND odi.issue_id=?)
          LEFT JOIN outlet_settlement_lines osl ON (i.source_type='CONSIGNMENT_SETTLEMENT' AND osl.settlement_id=i.source_id AND osl.issue_id=?)
          WHERE i.status <> 'VOID' AND (odi.issue_id IS NOT NULL OR osl.issue_id IS NOT NULL)";
  if ($from_dt) { $sql .= " AND i.issued_at >= ?"; $params[] = $from_dt; }
  if ($to_dt)   { $sql .= " AND i.issued_at <= ?"; $params[] = $to_dt; }
  $sql .= " ORDER BY i.distributor_id, i.outlet_id, i.issued_at, i.id";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $inv_rows = $st->fetchAll(PDO::FETCH_ASSOC);

  $metrics['inv_count'] = count($inv_rows);
  foreach ($inv_rows as $ir) $metrics['inv_total'] += (float)$ir['total_amount'];

  // Allocate outlet payments to these invoices FIFO per (distributor,outlet)
  // Payments are stored per outlet+distributor.
  $pairs = [];
  foreach ($inv_rows as $ir) {
    $k = ((int)$ir['distributor_id']).':'.((int)$ir['outlet_id']);
    if (!isset($pairs[$k])) $pairs[$k] = ['dist'=>(int)$ir['distributor_id'], 'out'=>(int)$ir['outlet_id']];
  }

  $paid_outlets_total = 0.0;
  $paid_office_total = 0.0;
  $due_total = 0.0;

  foreach ($pairs as $k => $pair) {
    $dist = $pair['dist'];
    $out = $pair['out'];

    // invoices for this pair
    $pair_invoices = [];
    foreach ($inv_rows as $ir) {
      if ((int)$ir['distributor_id']===$dist && (int)$ir['outlet_id']===$out) $pair_invoices[] = $ir;
    }

    // payments for this pair
    $params_p = [$out, $dist];
    $sql_p = "SELECT amount, status, paid_at FROM payments WHERE outlet_id=? AND distributor_id=? AND status IN ('HELD','POSTED') AND status <> 'VOID'";
    if ($from_dt) { $sql_p .= " AND paid_at >= ?"; $params_p[] = $from_dt; }
    if ($to_dt)   { $sql_p .= " AND paid_at <= ?"; $params_p[] = $to_dt; }
    $sql_p .= " ORDER BY paid_at, id";
    $st = $pdo->prepare($sql_p);
    $st->execute($params_p);
    $pay_rows = $st->fetchAll(PDO::FETCH_ASSOC);

    $paid_all = 0.0;
    $paid_posted = 0.0;
    foreach ($pay_rows as $pr) {
      $paid_all += (float)$pr['amount'];
      if ($pr['status'] === 'POSTED') $paid_posted += (float)$pr['amount'];
    }

    // FIFO allocate
    $rem_all = $paid_all;
    $rem_posted = $paid_posted;

    foreach ($pair_invoices as $ir) {
      $amt = (float)$ir['total_amount'];

      // allocation for collected-from-outlets
      $rem_inv = $amt;
      if ($rem_all > 0) {
        $use = min($rem_inv, $rem_all);
        $rem_inv -= $use;
        $rem_all -= $use;
      }

      // allocation for office posted (separately)
      $rem_inv2 = $amt;
      if ($rem_posted > 0) {
        $use2 = min($rem_inv2, $rem_posted);
        $rem_inv2 -= $use2;
        $rem_posted -= $use2;
      }

      $due_total += max(0.0, $rem_inv);
      $paid_outlets_total += ($amt - max(0.0, $rem_inv));
      $paid_office_total += ($amt - max(0.0, $rem_inv2));
    }
  }

  $metrics['paid_outlets'] = $paid_outlets_total;
  $metrics['paid_office'] = $paid_office_total;
  $metrics['inv_due'] = $due_total;
}

?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>ملخص الإصدار — MGZ</title>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
<style>
  body{font-family:'Tajawal',sans-serif;background:#f4f7f6;color:#222;padding:18px}
  .page-header{background:linear-gradient(135deg,#0d6efd,#6610f2);color:#fff;padding:18px 22px;border-radius:12px;margin-bottom:16px;display:flex;justify-content:space-between;align-items:center;gap:10px;flex-wrap:wrap}
  .page-header h2{margin:0;font-size:1.2rem}
  .card{background:#fff;border-radius:12px;box-shadow:0 2px 8px rgba(0,0,0,.08);margin-bottom:16px;overflow:hidden}
  .card-header{padding:12px 16px;border-bottom:1px solid #eee;font-weight:700;background:#fafafa}
  .card-body{padding:16px}
  .filter-grid{display:grid;grid-template-columns:2fr 1fr 1fr auto;gap:12px;align-items:end}
  label{display:block;font-size:.85rem;color:#555;margin-bottom:6px;font-weight:600}
  input[type=date],select{width:100%;padding:8px 10px;border:1px solid #ddd;border-radius:8px;background:#fff}
  .btn{display:inline-flex;align-items:center;gap:6px;padding:8px 14px;border-radius:8px;border:0;cursor:pointer;font-weight:700}
  .btn-primary{background:#0d6efd;color:#fff}
  .btn-outline{background:#fff;border:1px solid #ddd;color:#333}
  .stats{display:grid;grid-template-columns:repeat(auto-fit,minmax(190px,1fr));gap:12px}
  .stat{background:#fff;border-radius:12px;box-shadow:0 2px 8px rgba(0,0,0,.08);padding:14px;text-align:center;border-top:4px solid #0d6efd}
  .stat .num{font-size:1.4rem;font-weight:800}
  .stat .lbl{color:#777;margin-top:4px;font-size:.85rem}
  table{width:100%;border-collapse:collapse}
  th,td{padding:10px 12px;border-bottom:1px solid #eee;white-space:nowrap;text-align:right}
  th{background:#f8f9fa;font-size:.85rem;color:#555}
  .money{font-weight:800}
  .muted{color:#777;font-size:.85rem}
  @media(max-width:860px){.filter-grid{grid-template-columns:1fr}}
</style>
</head>
<body>
<a href="../dashboard.php" class="back-link no-print">&#8594; العودة للرئيسية</a>

<div class="page-header">
  <div>
    <h2>🧾 ملخص الإصدار</h2>
    <small>لوحة إغلاق الإصدار للمحاسب — <?= h($now) ?></small>
  </div>
  <div class="muted">اختر إصدارًا لعرض ملخص الكميات + المال</div>
</div>

<div class="card no-print">
  <div class="card-header">🔍 فلتر الملخص</div>
  <div class="card-body">
    <form method="get">
      <div class="filter-grid">
        <div>
          <label>الإصدار</label>
          <select name="issue_id" required>
            <option value="0">— اختر إصدار —</option>
            <?php foreach($issues as $it): ?>
              <option value="<?= (int)$it['id'] ?>" <?= $issue_id==(int)$it['id']?'selected':'' ?>><?= h(issue_label($it)) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>من تاريخ</label>
          <input type="date" name="from" value="<?= h($from_date) ?>">
        </div>
        <div>
          <label>إلى تاريخ</label>
          <input type="date" name="to" value="<?= h($to_date) ?>">
        </div>
        <div style="display:flex;gap:8px;align-items:end">
          <button class="btn btn-primary" type="submit">عرض</button>
          <a class="btn btn-outline" href="report_issue_summary.php">مسح</a>
        </div>
      </div>
    </form>
  </div>
</div>

<?php if ($issue_id <= 0): ?>
  <div class="card"><div class="card-body muted">اختر إصدارًا لعرض الملخص.</div></div>
<?php else: ?>
  <div class="stats">
    <div class="stat" style="border-top-color:#1a73e8">
      <div class="num"><?= (int)$metrics['whr_in'] ?></div>
      <div class="lbl">وارد المخزن (WHR)</div>
    </div>
    <div class="stat" style="border-top-color:#0b7285">
      <div class="num"><?= (int)$metrics['dis_out'] ?></div>
      <div class="lbl">صادر للموزعين (DIS)</div>
    </div>
    <div class="stat" style="border-top-color:#845ef7">
      <div class="num"><?= (int)$metrics['wrt_in'] ?></div>
      <div class="lbl">مرتجع للمخزن (WRT)</div>
    </div>
    <div class="stat" style="border-top-color:#34a853">
      <div class="num"><?= (int)$metrics['sold_direct_qty'] + (int)$metrics['sold_cons_qty'] ?></div>
      <div class="lbl">إجمالي المبيعات (عدد)</div>
    </div>
  </div>

  <div class="stats" style="margin-top:12px">
    <div class="stat" style="border-top-color:#34a853">
      <div class="num money"><?= fmt_omr($metrics['sold_direct_val'] + $metrics['sold_cons_val']) ?> ر.ع</div>
      <div class="lbl">إجمالي المبيعات (قيمة محاسبية)</div>
    </div>
    <div class="stat" style="border-top-color:#1a73e8">
      <div class="num money"><?= fmt_omr($metrics['paid_outlets']) ?> ر.ع</div>
      <div class="lbl">مدفوع من المنافذ للموزعين</div>
    </div>
    <div class="stat" style="border-top-color:#0c8599">
      <div class="num money"><?= fmt_omr($metrics['paid_office']) ?> ر.ع</div>
      <div class="lbl">مُعتمد بالمكتب (استلم المكتب)</div>
    </div>
    <div class="stat" style="border-top-color:#ea4335">
      <div class="num money"><?= fmt_omr($metrics['inv_due']) ?> ر.ع</div>
      <div class="lbl">المتبقي على المنافذ (لفواتير الإصدار)</div>
    </div>
  </div>

  <div class="card">
    <div class="card-header">📦 تفاصيل سريعة</div>
    <div class="card-body">
      <div style="overflow:auto">
        <table>
          <thead>
            <tr>
              <th>البند</th>
              <th>عدد</th>
              <th>قيمة (ر.ع.)</th>
            </tr>
          </thead>
          <tbody>
            <tr>
              <td>مبيعات بيع مباشر (Direct)</td>
              <td><?= (int)$metrics['sold_direct_qty'] ?></td>
              <td class="money"><?= fmt_omr($metrics['sold_direct_val']) ?></td>
            </tr>
            <tr>
              <td>مبيعات تسويات أمانة (Sold)</td>
              <td><?= (int)$metrics['sold_cons_qty'] ?></td>
              <td class="money"><?= fmt_omr($metrics['sold_cons_val']) ?></td>
            </tr>
            <tr>
              <td>تالف معلّق</td>
              <td><?= (int)$metrics['dam_pending'] ?></td>
              <td class="money"><?= fmt_omr(((float)($issue['cover_price'] ?? 0)) * (int)$metrics['dam_pending']) ?></td>
            </tr>
            <tr>
              <td>تالف معتمد</td>
              <td><?= (int)$metrics['dam_approved'] ?></td>
              <td class="money"><?= fmt_omr(((float)($issue['cover_price'] ?? 0)) * (int)$metrics['dam_approved']) ?></td>
            </tr>
            <tr>
              <td>تالف مرفوض</td>
              <td><?= (int)$metrics['dam_rejected'] ?></td>
              <td class="money"><?= fmt_omr(((float)($issue['cover_price'] ?? 0)) * (int)$metrics['dam_rejected']) ?></td>
            </tr>
            <tr>
              <td>عدد فواتير الإصدار</td>
              <td><?= (int)$metrics['inv_count'] ?></td>
              <td class="money"><?= fmt_omr($metrics['inv_total']) ?></td>
            </tr>
          </tbody>
        </table>
      </div>

      <div class="muted" style="margin-top:10px">
        * المبيعات المحاسبية = مجموع (qty × unit_price_applied) لبيع مباشر + تسويات أمانة (SUBMITTED/APPROVED).<br>
        * التحصيلات تُحتسب من المدفوعات (HELD + POSTED) المربوطة بفواتير هذا الإصدار (توزيع FIFO لكل منفذ/موزع).
      </div>
    </div>
  </div>
<?php endif; ?>

</body>
</html>
