<?php
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';

require_role(['ADMIN']);

$pdo = db();

function _parse_date_input($s) {
  $s = trim((string)$s);
  if ($s === '') return null;
  foreach (['Y-m-d', 'm/d/Y', 'd/m/Y'] as $fmt) {
    $d = DateTime::createFromFormat($fmt, $s);
    if ($d && $d->format($fmt) === $s) return $d;
  }
  return null;
}

$from_raw = $_GET['from'] ?? '';
$to_raw   = $_GET['to'] ?? '';

$from_d = _parse_date_input($from_raw);
$to_d   = _parse_date_input($to_raw);

$from_dt = $from_d ? ($from_d->format('Y-m-d') . ' 00:00:00') : null;
$to_dt   = $to_d   ? ($to_d->format('Y-m-d')   . ' 23:59:59') : null;

$from_date = $from_d ? $from_d->format('Y-m-d') : '';
$to_date   = $to_d   ? $to_d->format('Y-m-d')   : '';

function fmt3($n) { return number_format((float)$n, 3, '.', ''); }
function fmt_omr($n) {
  $n = (float)$n;
  // trim trailing zeros
  $s = number_format($n, 3, '.', '');
  $s = rtrim(rtrim($s, '0'), '.');
  return $s === '' ? '0' : $s;
}

$issue_id = (int)($_GET['issue_id'] ?? 0);
$distributor_id = (int)($_GET['distributor_id'] ?? 0);
$outlet_id = (int)($_GET['outlet_id'] ?? 0);
$entity_type = trim((string)($_GET['entity_type'] ?? ''));
$ref_type = trim((string)($_GET['ref_type'] ?? ''));

function issue_label($it) {
  $product = trim((string)($it['product_name'] ?? ''));
  $title = trim((string)($it['title'] ?? ''));
  $base = $product !== '' ? $product : ($title !== '' ? $title : 'إصدار');
  $m = (int)($it['issue_month'] ?? 0);
  $y = (int)($it['issue_year'] ?? 0);
  $datePart = ($m > 0 && $y > 0) ? (' - ' . str_pad((string)$m, 2, '0', STR_PAD_LEFT) . '/' . $y) : '';
  $num = $it['issue_number'] ?? null;
  $numPart = ($num !== null && $num !== '') ? (' - عدد ' . (int)$num) : '';
  return $base . $datePart . $numPart;
}

$issues = $pdo->query("SELECT i.id, p.name AS product_name, i.title, i.issue_month, i.issue_year, i.issue_number FROM issues i JOIN products p ON p.id=i.product_id ORDER BY i.issue_year DESC, i.issue_month DESC, i.id DESC")->fetchAll(PDO::FETCH_ASSOC);
$distributors = $pdo->query("SELECT id, name FROM distributors ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
$outlets = $pdo->query("SELECT id, name FROM outlets ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

$where = "l.status='POSTED'";
$params = [];

if ($issue_id > 0) { $where .= " AND l.issue_id=?"; $params[] = $issue_id; }
if ($entity_type !== '' && in_array($entity_type, ['WAREHOUSE','DISTRIBUTOR','OUTLET'], true)) {
  $where .= " AND l.entity_type=?"; $params[] = $entity_type;
}
if ($distributor_id > 0) { $where .= " AND l.entity_type='DISTRIBUTOR' AND l.entity_id=?"; $params[] = $distributor_id; }
if ($outlet_id > 0)      { $where .= " AND l.entity_type='OUTLET' AND l.entity_id=?";      $params[] = $outlet_id; }
if ($ref_type !== '')    { $where .= " AND l.ref_type=?"; $params[] = $ref_type; }

if ($from_dt) { $where .= " AND l.created_at >= ?"; $params[] = $from_dt; }
if ($to_dt)   { $where .= " AND l.created_at <= ?"; $params[] = $to_dt; }

$sql = "SELECT
  l.*,
  iss.title AS issue_title, iss.issue_month, iss.issue_year, iss.cover_price,
  d.name AS distributor_name,
  o.name AS outlet_name
FROM inventory_ledger l
JOIN issues iss ON iss.id = l.issue_id
LEFT JOIN distributors d ON (l.entity_type='DISTRIBUTOR' AND d.id = l.entity_id)
LEFT JOIN outlets o ON (l.entity_type='OUTLET' AND o.id = l.entity_id)
WHERE $where
ORDER BY l.created_at DESC, l.id DESC
LIMIT 2000";
$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

// Prefetch unit prices for deliveries and settlements
$delivery_ids = [];
$settlement_ids = [];
foreach ($rows as $r) {
  if ($r['ref_type'] === 'OUTLET_DELIVERY') $delivery_ids[(int)$r['ref_id']] = true;
  if ($r['ref_type'] === 'SETTLEMENT') $settlement_ids[(int)$r['ref_id']] = true;
}

$delivery_price = []; // [delivery_id][issue_id] => unit_price_applied
if ($delivery_ids) {
  $ids = array_keys($delivery_ids);
  $in = implode(',', array_fill(0, count($ids), '?'));
  $q = "SELECT delivery_id, issue_id, unit_price_applied FROM outlet_delivery_items WHERE delivery_id IN ($in)";
  $st = $pdo->prepare($q);
  $st->execute($ids);
  while ($row = $st->fetch(PDO::FETCH_ASSOC)) {
    $did = (int)$row['delivery_id']; $iid = (int)$row['issue_id'];
    if (!isset($delivery_price[$did])) $delivery_price[$did] = [];
    $delivery_price[$did][$iid] = (float)$row['unit_price_applied'];
  }
}

$settlement_price = []; // [settlement_id][issue_id] => unit_price_applied
if ($settlement_ids) {
  $ids = array_keys($settlement_ids);
  $in = implode(',', array_fill(0, count($ids), '?'));
  $q = "SELECT settlement_id, issue_id, unit_price_applied FROM outlet_settlement_lines WHERE settlement_id IN ($in)";
  $st = $pdo->prepare($q);
  $st->execute($ids);
  while ($row = $st->fetch(PDO::FETCH_ASSOC)) {
    $sid = (int)$row['settlement_id']; $iid = (int)$row['issue_id'];
    if (!isset($settlement_price[$sid])) $settlement_price[$sid] = [];
    $settlement_price[$sid][$iid] = (float)$row['unit_price_applied'];
  }
}

$now = (new DateTime('now'))->format('d/m/Y H:i');

// ref types list
$ref_types = ['' => '— الكل —', 'WH_RECEIPT'=>'WH_RECEIPT', 'DIST_ORDER'=>'DIST_ORDER', 'OUTLET_DELIVERY'=>'OUTLET_DELIVERY', 'SETTLEMENT'=>'SETTLEMENT', 'WRT'=>'WRT'];

?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>دفتر الأستاذ — MGZ</title>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
<style>
  body{font-family:'Tajawal',sans-serif;background:#f4f7f6;color:#222;padding:18px}
  .page-header{background:linear-gradient(135deg,#0d6efd,#6610f2);color:#fff;padding:18px 22px;border-radius:12px;margin-bottom:16px;display:flex;justify-content:space-between;align-items:center;gap:10px;flex-wrap:wrap}
  .page-header h2{margin:0;font-size:1.2rem}
  .card{background:#fff;border-radius:12px;box-shadow:0 2px 8px rgba(0,0,0,.08);margin-bottom:16px;overflow:hidden}
  .card-header{padding:12px 16px;border-bottom:1px solid #eee;font-weight:700;background:#fafafa}
  .card-body{padding:16px}
  .filter-grid{display:grid;grid-template-columns:2fr 2fr 2fr 1fr 1fr 1fr auto;gap:10px;align-items:end}
  label{display:block;font-size:.85rem;color:#555;margin-bottom:6px;font-weight:600}
  input[type=date],select{width:100%;padding:8px 10px;border:1px solid #ddd;border-radius:8px;background:#fff}
  .btn{display:inline-flex;align-items:center;gap:6px;padding:8px 14px;border-radius:8px;border:0;cursor:pointer;font-weight:700}
  .btn-primary{background:#0d6efd;color:#fff}
  .btn-outline{background:#fff;border:1px solid #ddd;color:#333}
  table{width:100%;border-collapse:collapse}
  th,td{padding:10px 12px;border-bottom:1px solid #eee;white-space:nowrap;text-align:right}
  th{background:#f8f9fa;font-size:.85rem;color:#555}
  .muted{color:#777;font-size:.85rem}
  .money{font-weight:800}
  .tag{display:inline-block;padding:2px 10px;border-radius:999px;background:#eef2ff;color:#1a73e8;font-weight:800;font-size:.75rem}
  @media(max-width:1100px){.filter-grid{grid-template-columns:1fr}}
</style>
</head>
<body>
<a href="../dashboard.php" class="back-link no-print">&#8594; العودة للرئيسية</a>

<div class="page-header">
  <div>
    <h2>📒 دفتر الأستاذ للحركات</h2>
    <small>حركات المخزون + الربط بالمراجع — <?= h($now) ?></small>
  </div>
  <div class="muted">* القيمة تقديرية حسب السعر بعد الخصم إن وُجد</div>
</div>

<div class="card no-print">
  <div class="card-header">🔍 فلتر التقرير</div>
  <div class="card-body">
    <form method="get">
      <div class="filter-grid">
        <div>
          <label>الإصدار</label>
          <select name="issue_id">
            <option value="0">— جميع الإصدارات —</option>
            <?php foreach($issues as $it): ?>
              <option value="<?= (int)$it['id'] ?>" <?= $issue_id==(int)$it['id']?'selected':'' ?>><?= h(issue_label($it)) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>الطرف</label>
          <select name="entity_type">
            <option value="">— الكل —</option>
            <option value="WAREHOUSE" <?= $entity_type==='WAREHOUSE'?'selected':'' ?>>المخزن الرئيسي</option>
            <option value="DISTRIBUTOR" <?= $entity_type==='DISTRIBUTOR'?'selected':'' ?>>موزع</option>
            <option value="OUTLET" <?= $entity_type==='OUTLET'?'selected':'' ?>>منفذ</option>
          </select>
        </div>
        <div>
          <label>موزع</label>
          <select name="distributor_id">
            <option value="0">— الكل —</option>
            <?php foreach($distributors as $d): ?>
              <option value="<?= (int)$d['id'] ?>" <?= $distributor_id==(int)$d['id']?'selected':'' ?>><?= h($d['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>منفذ</label>
          <select name="outlet_id">
            <option value="0">— الكل —</option>
            <?php foreach($outlets as $o): ?>
              <option value="<?= (int)$o['id'] ?>" <?= $outlet_id==(int)$o['id']?'selected':'' ?>><?= h($o['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>نوع المرجع</label>
          <select name="ref_type">
            <?php foreach($ref_types as $k=>$v): ?>
              <option value="<?= h($k) ?>" <?= $ref_type===$k?'selected':'' ?>><?= h($v) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>من</label>
          <input type="date" name="from" value="<?= h($from_date) ?>">
        </div>
        <div>
          <label>إلى</label>
          <input type="date" name="to" value="<?= h($to_date) ?>">
        </div>
        <div style="display:flex;gap:8px;align-items:end">
          <button class="btn btn-primary" type="submit">تطبيق</button>
          <a class="btn btn-outline" href="report_ledger.php">مسح</a>
        </div>
      </div>
    </form>
  </div>
</div>

<div class="card">
  <div class="card-header">📄 النتائج (آخر 2000 حركة)</div>
  <div class="card-body">
    <div style="overflow:auto">
      <table>
        <thead>
          <tr>
            <th>التاريخ</th>
            <th>الطرف</th>
            <th>الإصدار</th>
            <th>Bucket</th>
            <th>داخل</th>
            <th>خارج</th>
            <th>نوع المرجع</th>
            <th>المرجع</th>
            <th>سعر/وحدة</th>
            <th>القيمة</th>
            <th>ملاحظة</th>
          </tr>
        </thead>
        <tbody>
          <?php if(!$rows): ?>
            <tr><td colspan="11" class="muted">لا توجد بيانات</td></tr>
          <?php else: ?>
            <?php foreach($rows as $r):
              $entity = '';
              if ($r['entity_type'] === 'WAREHOUSE') $entity = 'المخزن الرئيسي';
              elseif ($r['entity_type'] === 'DISTRIBUTOR') $entity = 'موزع: ' . ($r['distributor_name'] ?? ('#'.$r['entity_id']));
              elseif ($r['entity_type'] === 'OUTLET') $entity = 'منفذ: ' . ($r['outlet_name'] ?? ('#'.$r['entity_id']));
              else $entity = $r['entity_type'] . ' #' . $r['entity_id'];

              $issue_lbl = ($r['issue_title'] ?? '') . ' - ' . str_pad((string)$r['issue_month'],2,'0',STR_PAD_LEFT) . '/' . $r['issue_year'];

              $qty = abs((int)$r['qty_in'] - (int)$r['qty_out']);
              $unit = null;
              if ($r['ref_type'] === 'OUTLET_DELIVERY') {
                $did = (int)$r['ref_id']; $iid=(int)$r['issue_id'];
                if (isset($delivery_price[$did]) && isset($delivery_price[$did][$iid])) $unit = (float)$delivery_price[$did][$iid];
              } elseif ($r['ref_type'] === 'SETTLEMENT') {
                $sid = (int)$r['ref_id']; $iid=(int)$r['issue_id'];
                if (isset($settlement_price[$sid]) && isset($settlement_price[$sid][$iid])) $unit = (float)$settlement_price[$sid][$iid];
              }
              if ($unit === null) $unit = (float)$r['cover_price'];
              $val = $unit * $qty;

              ?>
              <tr>
                <td><?= h($r['created_at']) ?></td>
                <td><?= h($entity) ?></td>
                <td><?= h($issue_lbl) ?></td>
                <td><span class="tag"><?= h($r['bucket']) ?></span></td>
                <td><?= (int)$r['qty_in'] ?></td>
                <td><?= (int)$r['qty_out'] ?></td>
                <td><?= h($r['ref_type']) ?></td>
                <td>#<?= (int)$r['ref_id'] ?> / <?= h($r['ref_line']) ?></td>
                <td class="money"><?= fmt_omr($unit) ?></td>
                <td class="money"><?= fmt_omr($val) ?></td>
                <td class="muted"><?= h($r['note'] ?? '') ?></td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="muted" style="margin-top:10px">
      * قيمة الصف = |داخل-خارج| × سعر/وحدة (يُستخدم السعر بعد الخصم إن وجد في المرجع).
    </div>
  </div>
</div>

</body>
</html>
