<?php
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';

require_role(['ADMIN']);

$pdo = db();

function _parse_date_input($s) {
  $s = trim((string)$s);
  if ($s === '') return null;
  foreach (['Y-m-d', 'm/d/Y', 'd/m/Y'] as $fmt) {
    $d = DateTime::createFromFormat($fmt, $s);
    if ($d && $d->format($fmt) === $s) return $d;
  }
  return null;
}

$from_raw = $_GET['from'] ?? '';
$to_raw   = $_GET['to'] ?? '';

$from_d = _parse_date_input($from_raw);
$to_d   = _parse_date_input($to_raw);

$from_dt = $from_d ? ($from_d->format('Y-m-d') . ' 00:00:00') : null;
$to_dt   = $to_d   ? ($to_d->format('Y-m-d')   . ' 23:59:59') : null;

$from_date = $from_d ? $from_d->format('Y-m-d') : '';
$to_date   = $to_d   ? $to_d->format('Y-m-d')   : '';

function fmt3($n) { return number_format((float)$n, 3, '.', ''); }
function fmt_omr($n) {
  $n = (float)$n;
  // trim trailing zeros
  $s = number_format($n, 3, '.', '');
  $s = rtrim(rtrim($s, '0'), '.');
  return $s === '' ? '0' : $s;
}

$outlet_id = (int)($_GET['outlet_id'] ?? 0);
$distributor_id = (int)($_GET['distributor_id'] ?? 0);

$outlets = $pdo->query("SELECT id, name FROM outlets ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
$distributors = $pdo->query("SELECT id, name FROM distributors ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

$where_inv = "i.status <> 'VOID'";
$params_inv = [];
if ($outlet_id > 0) { $where_inv .= " AND i.outlet_id = ?"; $params_inv[] = $outlet_id; }
if ($distributor_id > 0) { $where_inv .= " AND i.distributor_id = ?"; $params_inv[] = $distributor_id; }
if ($from_dt) { $where_inv .= " AND i.issued_at >= ?"; $params_inv[] = $from_dt; }
if ($to_dt)   { $where_inv .= " AND i.issued_at <= ?"; $params_inv[] = $to_dt; }

$inv_sql = "SELECT i.id, i.invoice_no, i.outlet_id, i.distributor_id, i.issued_at, i.total_amount
            FROM invoices i
            WHERE $where_inv
            ORDER BY i.outlet_id, i.issued_at, i.id";
$st = $pdo->prepare($inv_sql);
$st->execute($params_inv);
$invoices = $st->fetchAll(PDO::FETCH_ASSOC);

// Payments (money paid by outlet to distributor), include HELD+POSTED, exclude VOID
$where_pay = "p.status IN ('HELD','POSTED') AND p.outlet_id IS NOT NULL";
$params_pay = [];
if ($outlet_id > 0) { $where_pay .= " AND p.outlet_id = ?"; $params_pay[] = $outlet_id; }
if ($distributor_id > 0) { $where_pay .= " AND p.distributor_id = ?"; $params_pay[] = $distributor_id; }
if ($from_dt) { $where_pay .= " AND p.paid_at >= ?"; $params_pay[] = $from_dt; }
if ($to_dt)   { $where_pay .= " AND p.paid_at <= ?"; $params_pay[] = $to_dt; }

$pay_sql = "SELECT p.id, p.outlet_id, p.distributor_id, p.paid_at, p.amount
            FROM payments p
            WHERE $where_pay
            ORDER BY p.outlet_id, p.paid_at, p.id";
$st = $pdo->prepare($pay_sql);
$st->execute($params_pay);
$payments = $st->fetchAll(PDO::FETCH_ASSOC);

// Group
$inv_by_outlet = [];
foreach ($invoices as $inv) {
  $oid = (int)$inv['outlet_id'];
  if (!isset($inv_by_outlet[$oid])) $inv_by_outlet[$oid] = [];
  $inv_by_outlet[$oid][] = $inv;
}

$pay_by_outlet = [];
foreach ($payments as $p) {
  $oid = (int)$p['outlet_id'];
  if (!isset($pay_by_outlet[$oid])) $pay_by_outlet[$oid] = 0.0;
  $pay_by_outlet[$oid] += (float)$p['amount'];
}

// helper for name maps
$outlet_map = [];
foreach ($outlets as $o) $outlet_map[(int)$o['id']] = $o['name'];

$dist_map = [];
foreach ($distributors as $d) $dist_map[(int)$d['id']] = $d['name'];

$today = new DateTime('now');
$now = $today->format('d/m/Y H:i');

$rows = [];
$tot_invoiced = 0.0; $tot_paid = 0.0; $tot_due = 0.0;
$tot_b0 = 0.0; $tot_b1 = 0.0; $tot_b2 = 0.0;

foreach ($inv_by_outlet as $oid => $inv_list) {
  $paid = isset($pay_by_outlet[$oid]) ? (float)$pay_by_outlet[$oid] : 0.0;

  // FIFO allocate payments to invoices
  $remaining_paid = $paid;
  $b0 = 0.0; $b1 = 0.0; $b2 = 0.0;
  $invoiced = 0.0;
  foreach ($inv_list as $inv) {
    $amt = (float)$inv['total_amount'];
    $invoiced += $amt;
    $rem = $amt;
    if ($remaining_paid > 0) {
      $use = min($rem, $remaining_paid);
      $rem -= $use;
      $remaining_paid -= $use;
    }
    if ($rem > 0.00001) {
      $issued = new DateTime($inv['issued_at']);
      $days = (int)$issued->diff($today)->format('%a');
      if ($days <= 7) $b0 += $rem;
      else if ($days <= 30) $b1 += $rem;
      else $b2 += $rem;
    }
  }
  $due = max(0.0, $invoiced - $paid);

  $rows[] = [
    'outlet_id' => $oid,
    'outlet_name' => $outlet_map[$oid] ?? ('#'.$oid),
    'invoiced' => $invoiced,
    'paid' => $paid,
    'due' => $due,
    'b0' => $b0,
    'b1' => $b1,
    'b2' => $b2,
  ];

  $tot_invoiced += $invoiced;
  $tot_paid += $paid;
  $tot_due += $due;
  $tot_b0 += $b0; $tot_b1 += $b1; $tot_b2 += $b2;
}

// sort by due desc
usort($rows, function($a,$b){
  if ($a['due']==$b['due']) return 0;
  return ($a['due'] < $b['due']) ? 1 : -1;
});

?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>مديونيات المنافذ — MGZ</title>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
<style>
  body{font-family:'Tajawal',sans-serif;background:#f4f7f6;color:#222;padding:18px}
  .page-header{background:linear-gradient(135deg,#0d6efd,#6610f2);color:#fff;padding:18px 22px;border-radius:12px;margin-bottom:16px;display:flex;justify-content:space-between;align-items:center;gap:10px;flex-wrap:wrap}
  .page-header h2{margin:0;font-size:1.2rem}
  .card{background:#fff;border-radius:12px;box-shadow:0 2px 8px rgba(0,0,0,.08);margin-bottom:16px;overflow:hidden}
  .card-header{padding:12px 16px;border-bottom:1px solid #eee;font-weight:700;background:#fafafa}
  .card-body{padding:16px}
  .filter-grid{display:grid;grid-template-columns:2fr 2fr 1fr 1fr auto;gap:12px;align-items:end}
  label{display:block;font-size:.85rem;color:#555;margin-bottom:6px;font-weight:600}
  input[type=date],select{width:100%;padding:8px 10px;border:1px solid #ddd;border-radius:8px;background:#fff}
  .btn{display:inline-flex;align-items:center;gap:6px;padding:8px 14px;border-radius:8px;border:0;cursor:pointer;font-weight:700}
  .btn-primary{background:#0d6efd;color:#fff}
  .btn-outline{background:#fff;border:1px solid #ddd;color:#333}
  .stats{display:grid;grid-template-columns:repeat(auto-fit,minmax(180px,1fr));gap:12px}
  .stat{background:#fff;border-radius:12px;box-shadow:0 2px 8px rgba(0,0,0,.08);padding:14px;text-align:center;border-top:4px solid #0d6efd}
  .stat .num{font-size:1.4rem;font-weight:800}
  .stat .lbl{color:#777;margin-top:4px;font-size:.85rem}
  table{width:100%;border-collapse:collapse}
  th,td{padding:10px 12px;border-bottom:1px solid #eee;white-space:nowrap;text-align:right}
  th{background:#f8f9fa;font-size:.85rem;color:#555}
  .money{font-weight:800}
  .muted{color:#777;font-size:.85rem}
  .bad{color:#ea4335}
  @media(max-width:860px){.filter-grid{grid-template-columns:1fr}}
</style>
</head>
<body>
<a href="../dashboard.php" class="back-link no-print">&#8594; العودة للرئيسية</a>

<div class="page-header">
  <div>
    <h2>📌 تقرير مديونيات المنافذ</h2>
    <small>تقرير مستحقات المنافذ (المدفوع للموزع) — <?= h($now) ?></small>
  </div>
  <div class="muted">عمر المديونية محسوب من تاريخ الفاتورة (issued_at)</div>
</div>

<div class="card no-print">
  <div class="card-header">🔍 فلتر التقرير</div>
  <div class="card-body">
    <form method="get">
      <div class="filter-grid">
        <div>
          <label>المنفذ</label>
          <select name="outlet_id">
            <option value="0">— جميع المنافذ —</option>
            <?php foreach($outlets as $o): ?>
              <option value="<?= (int)$o['id'] ?>" <?= $outlet_id==(int)$o['id']?'selected':'' ?>><?= h($o['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>الموزع (اختياري)</label>
          <select name="distributor_id">
            <option value="0">— جميع الموزعين —</option>
            <?php foreach($distributors as $d): ?>
              <option value="<?= (int)$d['id'] ?>" <?= $distributor_id==(int)$d['id']?'selected':'' ?>><?= h($d['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>من تاريخ</label>
          <input type="date" name="from" value="<?= h($from_date) ?>">
        </div>
        <div>
          <label>إلى تاريخ</label>
          <input type="date" name="to" value="<?= h($to_date) ?>">
        </div>
        <div style="display:flex;gap:8px;align-items:end">
          <button class="btn btn-primary" type="submit">تطبيق</button>
          <a class="btn btn-outline" href="report_outlet_arrears.php">مسح</a>
        </div>
      </div>
    </form>
  </div>
</div>

<div class="stats">
  <div class="stat" style="border-top-color:#1a73e8">
    <div class="num money"><?= fmt_omr($tot_invoiced) ?> ر.ع</div>
    <div class="lbl">إجمالي الفواتير</div>
  </div>
  <div class="stat" style="border-top-color:#34a853">
    <div class="num money"><?= fmt_omr($tot_paid) ?> ر.ع</div>
    <div class="lbl">إجمالي المدفوع للموزع</div>
  </div>
  <div class="stat" style="border-top-color:#ea4335">
    <div class="num money bad"><?= fmt_omr($tot_due) ?> ر.ع</div>
    <div class="lbl">إجمالي المتبقي على المنافذ</div>
  </div>
</div>

<div class="card">
  <div class="card-header">📄 النتائج</div>
  <div class="card-body">
    <div style="overflow:auto">
      <table>
        <thead>
          <tr>
            <th>المنفذ</th>
            <th>إجمالي الفواتير</th>
            <th>مدفوع للموزع</th>
            <th>المتبقي</th>
            <th>0-7 أيام</th>
            <th>8-30 يوم</th>
            <th>31+ يوم</th>
          </tr>
        </thead>
        <tbody>
          <?php if(!$rows): ?>
            <tr><td colspan="7" class="muted">لا توجد بيانات</td></tr>
          <?php else: ?>
            <?php foreach($rows as $r): ?>
              <tr>
                <td><?= h($r['outlet_name']) ?></td>
                <td class="money"><?= fmt_omr($r['invoiced']) ?> ر.ع</td>
                <td class="money"><?= fmt_omr($r['paid']) ?> ر.ع</td>
                <td class="money bad"><?= fmt_omr($r['due']) ?> ر.ع</td>
                <td class="money"><?= fmt_omr($r['b0']) ?> ر.ع</td>
                <td class="money"><?= fmt_omr($r['b1']) ?> ر.ع</td>
                <td class="money"><?= fmt_omr($r['b2']) ?> ر.ع</td>
              </tr>
            <?php endforeach; ?>
            <tr style="background:#f0f4ff;font-weight:800">
              <td>الإجمالي</td>
              <td><?= fmt_omr($tot_invoiced) ?> ر.ع</td>
              <td><?= fmt_omr($tot_paid) ?> ر.ع</td>
              <td><?= fmt_omr($tot_due) ?> ر.ع</td>
              <td><?= fmt_omr($tot_b0) ?> ر.ع</td>
              <td><?= fmt_omr($tot_b1) ?> ر.ع</td>
              <td><?= fmt_omr($tot_b2) ?> ر.ع</td>
            </tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="muted" style="margin-top:10px">
      * المدفوع للموزع = مجموع المدفوعات (HELD + POSTED) — لا يشترط اعتماد المكتب.
    </div>
  </div>
</div>

</body>
</html>
