<?php
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/csrf.php';
require_once __DIR__ . '/../app/docno.php';
require_once __DIR__ . '/../app/upload.php';
require_once __DIR__ . '/../app/ledger.php';
require_once __DIR__ . '/../app/audit.php';

$me  = require_role(['ADMIN','DISTRIBUTOR']);
$pdo = db();

// Ensure optional columns exist for richer damaged reporting (safe no-op if already present)
function ensure_damage_reports_columns(PDO $pdo): void {
  try {
    $cols = $pdo->query("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='damage_reports'")
                ->fetchAll(PDO::FETCH_COLUMN);
    $need = [];
    if (!in_array('settlement_id', $cols, true)) $need[] = "ADD COLUMN settlement_id BIGINT(20) UNSIGNED NULL AFTER outlet_id";
    if (!in_array('unit_price_applied', $cols, true)) $need[] = "ADD COLUMN unit_price_applied DECIMAL(10,3) NULL AFTER qty";
    if (!in_array('cover_price', $cols, true)) $need[] = "ADD COLUMN cover_price DECIMAL(10,3) NULL AFTER unit_price_applied";
    if ($need) {
      $pdo->exec("ALTER TABLE damage_reports " . implode(', ', $need));
    }
  } catch (Throwable $e) {
    // Don't block settlement if hosting disallows INFORMATION_SCHEMA/ALTER; fallback to old behavior.
  }
}
$csrf = csrf_token();
$action = $_GET['action'] ?? 'list';

// ─────────────────────────────────────────────────────────────
// Admin: approve / reject settlement (manager decision)
// ─────────────────────────────────────────────────────────────
if ($me['role'] === 'ADMIN' && $_SERVER['REQUEST_METHOD'] === 'POST' && (($_POST['op'] ?? '') === 'settlement_decide')) {
  csrf_check();
  $sid = (int)($_POST['id'] ?? 0);
  $decision = $_POST['decision'] ?? '';
  $reason = trim($_POST['reason'] ?? '');
  if ($sid <= 0 || !in_array($decision, ['APPROVED','REJECTED'], true)) {
    flash('error', 'طلب غير صالح.');
    redirect('settlements.php');
  }

  try {
    $pdo->beginTransaction();

    // lock settlement row
    $pdo->prepare("SELECT id FROM outlet_settlements WHERE id=? FOR UPDATE")->execute([$sid]);

    // لا تعتمد التسوية إذا كان هناك تالف/مراجعات معلّقة مرتبطة بها
    $st = $pdo->prepare("SELECT COUNT(*) FROM damage_reports WHERE settlement_id=? AND status='PENDING'");
    $st->execute([$sid]);
    $pendingDamage = (int)$st->fetchColumn();

    if ($decision === 'APPROVED' && $pendingDamage > 0) {
      throw new RuntimeException('لا يمكن اعتماد التسوية قبل مراجعة جميع طلبات التالف المرتبطة بها.');
    }

    $st2 = $pdo->prepare("UPDATE outlet_settlements
                          SET status=?, manager_by=?, manager_at=NOW(), manager_reason=?
                          WHERE id=?");
    $st2->execute([$decision, (int)$me['id'], $reason !== '' ? $reason : null, $sid]);

    audit_log($pdo, (int)$me['id'], $decision === 'APPROVED' ? 'APPROVE' : 'REJECT', 'outlet_settlements', $sid, ['reason'=>$reason]);

    $pdo->commit();
    flash('success', $decision === 'APPROVED' ? 'تم اعتماد التسوية.' : 'تم رفض التسوية.');
  } catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    flash('error', 'تعذر تنفيذ العملية: ' . $e->getMessage());
  }
  redirect('settlements.php');
}



function json_out($data, int $code=200): void {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}

function my_distributor_id(array $me): ?int {
  if ($me['role'] === 'DISTRIBUTOR') {
    return !empty($me['distributor_id']) ? (int)$me['distributor_id'] : null;
  }
  return null;
}

function fetch_delivery(PDO $pdo, int $id, ?int $restrictDistributorId=null): ?array {
  $sql = "SELECT od.*, o.name AS outlet_name, d.name AS distributor_name
          FROM outlet_deliveries od
          JOIN outlets o ON o.id=od.outlet_id
          JOIN distributors d ON d.id=od.distributor_id
          WHERE od.id=? AND od.status='APPROVED' AND od.deal_type='CONSIGNMENT'";
  $args = [$id];
  if ($restrictDistributorId !== null) { $sql .= " AND od.distributor_id=?"; $args[] = $restrictDistributorId; }
  $st = $pdo->prepare($sql);
  $st->execute($args);
  $r = $st->fetch();
  return $r ?: null;
}

function fetch_delivery_items(PDO $pdo, int $deliveryId): array {
  // include issue_month/issue_year for display in modal + unit_price_applied from delivery line
  $st = $pdo->prepare("SELECT odi.*, p.name AS product_name, i.issue_number, i.issue_month, i.issue_year, i.cover_price
                       FROM outlet_delivery_items odi
                       JOIN issues i ON i.id=odi.issue_id
                       JOIN products p ON p.id=i.product_id
                       WHERE odi.delivery_id=?");
  $st->execute([$deliveryId]);
  return $st->fetchAll();
}

function distributor_issue_balances(PDO $pdo, int $distributorId, array $issueIds): array {
  $issueIds = array_values(array_filter(array_map('intval', $issueIds), fn($v)=>$v>0));
  if (!$issueIds) return [];

  // AVAILABLE from v_inventory_balance; HOLD from inventory_ledger directly (the view aggregates AVAILABLE only)
  $placeholders = implode(',', array_fill(0, count($issueIds), '?'));
  $argsAvail = array_merge([$distributorId], $issueIds);

  $stA = $pdo->prepare("SELECT issue_id, COALESCE(balance_qty,0) AS avail_qty
                        FROM v_inventory_balance
                        WHERE entity_type='DISTRIBUTOR' AND entity_id=? AND issue_id IN ($placeholders)");
  $stA->execute($argsAvail);
  $avail = [];
  foreach ($stA->fetchAll() as $r) {
    $avail[(int)$r['issue_id']] = (int)$r['avail_qty'];
  }

  $stH = $pdo->prepare("SELECT issue_id, COALESCE(SUM(qty_in - qty_out),0) AS hold_qty
                        FROM inventory_ledger
                        WHERE entity_type='DISTRIBUTOR' AND entity_id=? AND status='POSTED' AND bucket='HOLD' AND issue_id IN ($placeholders)
                        GROUP BY issue_id");
  $stH->execute($argsAvail);
  $hold = [];
  foreach ($stH->fetchAll() as $r) {
    $hold[(int)$r['issue_id']] = (int)$r['hold_qty'];
  }

  $out = [];
  foreach ($issueIds as $iid) {
    $out[(int)$iid] = ['avail'=>($avail[(int)$iid] ?? 0), 'hold'=>($hold[(int)$iid] ?? 0)];
  }
  return $out;
}

function settlement_exists(PDO $pdo, int $deliveryId): bool {
  // يعتمد على وجود العمود delivery_id في outlet_settlements
  try {
    $st = $pdo->prepare("SELECT COUNT(*) FROM outlet_settlements WHERE delivery_id=?");
    $st->execute([$deliveryId]);
    return (int)$st->fetchColumn() > 0;
  } catch (Throwable $e) {
    return false;
  }
}

function distributor_commission(PDO $pdo, int $distributorId, float $grossAmount): float {
  $st = $pdo->prepare("SELECT commission_type, commission_value FROM distributors WHERE id=?");
  $st->execute([$distributorId]);
  $d = $st->fetch();
  if (!$d) return 0.0;
  if ($d['commission_type'] === 'PERCENT') return round($grossAmount * max(0.0, (float)$d['commission_value']) / 100.0, 3);
  if ($d['commission_type'] === 'FIXED')   return round(max(0.0, (float)$d['commission_value']), 3);
  return 0.0;
}

$distId = my_distributor_id($me);
// ensure patch applied (delivery_id column)
$hasDeliveryCol = true;
try { $pdo->query("SELECT delivery_id FROM outlet_settlements LIMIT 1"); } catch (Throwable $e) { $hasDeliveryCol = false; }


// API: load delivery for modal
if ($action === 'api_delivery') {
  if (!$hasDeliveryCol) json_out(["ok"=>false,"error"=>"يجب تشغيل DB_PATCH_SETL.sql أولاً (إضافة delivery_id)"], 500);

  $id = (int)($_GET['id'] ?? 0);
  if ($id<=0) json_out(['ok'=>false,'error'=>'Bad id'], 400);
  $d = fetch_delivery($pdo, $id, $distId);
  if (!$d) json_out(['ok'=>false,'error'=>'Not found'], 404);
  if (settlement_exists($pdo, $id)) json_out(['ok'=>false,'error'=>'تمت تسوية هذه المعاملة بالفعل'], 409);
  $items = fetch_delivery_items($pdo, $id);
  // enrich with distributor current balances (AVAILABLE + HOLD)
  $issueIds = array_map(fn($it)=>(int)$it['issue_id'], $items);
  $bal = distributor_issue_balances($pdo, (int)$d['distributor_id'], $issueIds);
  foreach ($items as &$it) {
    $iid = (int)$it['issue_id'];
    $it['dist_avail_qty'] = $bal[$iid]['avail'] ?? 0;
    $it['dist_hold_qty']  = $bal[$iid]['hold'] ?? 0;
  }
  unset($it);
  json_out(['ok'=>true,'delivery'=>$d,'items'=>$items]);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  if (!consume_form_token("settlement")) { die("تم إرسال الطلب مسبقاً."); }
$op = $_POST['op'] ?? '';

  if ($op === 'settle_delivery') {
    if ($distId === null) exit('Only distributor');
    $delivery_id = (int)($_POST['delivery_id'] ?? 0);
    $cash_collected = (float)($_POST['cash_collected'] ?? 0);
    $cash_collected = max(0.0, $cash_collected);

    $d = fetch_delivery($pdo, $delivery_id, $distId);
    if (!$d) exit('Delivery not found');
    // NOTE: quick pre-check only; authoritative guard happens inside transaction with locks.
    if (settlement_exists($pdo, $delivery_id)) exit('Already settled');

    $issue_ids = $_POST['issue_id'] ?? [];
    $sold      = $_POST['qty_sold'] ?? [];
    $ret       = $_POST['qty_returned'] ?? [];
    $dam       = $_POST['qty_damaged'] ?? [];

    if (!is_array($issue_ids) || count($issue_ids) === 0) exit('No lines');

    // load delivered quantities + السعر المطبق في طلب التسليم الأصلي (يشمل الخصم)
    $delivered = [];
    $appliedUnit = [];
    foreach (fetch_delivery_items($pdo, $delivery_id) as $it) {
      $iid = (int)$it['issue_id'];
      $delivered[$iid] = (int)$it['qty_delivered'];
      $appliedUnit[$iid] = ($it['unit_price_applied'] !== null) ? (float)$it['unit_price_applied'] : (float)$it['cover_price'];
    }

    $lines = [];
    $hasDamage = false;
    for ($k=0; $k<count($issue_ids); $k++) {
      $iid = (int)$issue_ids[$k];
      if ($iid<=0) continue;
      $qs = (int)($sold[$k] ?? 0);
      $qr = (int)($ret[$k] ?? 0);
      $qd = (int)($dam[$k] ?? 0);
      if ($qs<0 || $qr<0 || $qd<0) exit('Bad qty');
      $orig = $delivered[$iid] ?? 0;
      if ($orig<=0) exit('Issue not in delivery');
      if (($qs+$qr+$qd) !== $orig) exit('يجب أن يكون (تحصيل + مرتجع + تالف) = الكمية الأصلية لكل سطر');
      if ($qd>0) $hasDamage = true;
      $lines[] = ['issue_id'=>$iid,'qty_sold'=>$qs,'qty_returned'=>$qr,'qty_damaged'=>$qd,'orig'=>$orig];
    }
    if (!$lines) exit('No valid lines');

    // damage photo required if any damage
    $damagePhotoPath = null;
    $damageAttId = null;
    if ($hasDamage) {
      if (empty($_FILES['damage_photo']) || !is_array($_FILES['damage_photo']) || ($_FILES['damage_photo']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
        exit('صورة التالف إجبارية عند وجود تالف');
      }
      // CHANGE START: store as attachment + protected download URL
      $damageAttId = save_upload($_FILES['damage_photo'], (int)$me['id'], 'attachments', [
        'entity_type' => 'DAMAGE_REPORT',
        'entity_id' => null,
        'distributor_id' => (int)$d['distributor_id'],
        'outlet_id' => (int)$d['outlet_id'],
      ]);
      $cfg = require __DIR__ . '/../app/config.php';
      $baseUrl = rtrim((string)($cfg['app']['base_url'] ?? ''), '/');
      $damagePhotoPath = $baseUrl . '/download.php?id=' . (int)$damageAttId;
      // CHANGE END
    }

    $pdo->beginTransaction();
    try {
      // CHANGE START: prevent duplicate settlement (race) by locking delivery + settlement scope
      $pdo->prepare("SELECT id FROM outlet_deliveries WHERE id=? FOR UPDATE")->execute([$delivery_id]);
      $pdo->prepare("SELECT id FROM outlet_settlements WHERE delivery_id=? FOR UPDATE")->execute([$delivery_id]);
      if (settlement_exists($pdo, $delivery_id)) {
        throw new RuntimeException('Already settled');
      }
      // CHANGE END

      $settleAt = date('Y-m-d H:i:s');
      $settleNo = next_doc_no($pdo, 'OST', date('Y-m-d'));
      $status = $hasDamage ? 'SUBMITTED' : 'APPROVED';

      // requires column delivery_id
      $pdo->prepare("INSERT INTO outlet_settlements (settlement_no, distributor_id, outlet_id, delivery_id, settlement_at, created_by, status)
                     VALUES (?,?,?,?,?,?,?)")
          ->execute([$settleNo, (int)$d['distributor_id'], (int)$d['outlet_id'], $delivery_id, $settleAt, (int)$me['id'], $status]);
      $settlementId = (int)$pdo->lastInsertId();

      // CHANGE START: link damage photo attachment to this settlement for download RBAC
      if (!empty($damageAttId)) {
        $pdo->prepare("UPDATE attachments SET entity_type='SETTLEMENT_DAMAGE', entity_id=?, distributor_id=?, outlet_id=? WHERE id=?")
            ->execute([$settlementId, (int)$d['distributor_id'], (int)$d['outlet_id'], (int)$damageAttId]);
      }
      // CHANGE END

      // create invoice for sold qty (always), even if damage pending
      $invNo = next_doc_no($pdo, 'INV', date('Y-m-d'));
      $pdo->prepare("INSERT INTO invoices (invoice_no, outlet_id, distributor_id, source_type, source_id, issued_at, status, currency, total_amount, total_paid, balance_due, created_by)
                     VALUES (?,?,?,'CONSIGNMENT_SETTLEMENT',?,NOW(),'ISSUED','OMR',0,0,0,?)")
          ->execute([$invNo, (int)$d['outlet_id'], (int)$d['distributor_id'], $settlementId, (int)$me['id']]);
      $invoiceId = (int)$pdo->lastInsertId();

      $total = 0.0;
      foreach ($lines as $ln) {
        $iid = (int)$ln['issue_id'];
        $qs = (int)$ln['qty_sold'];
        $qr = (int)$ln['qty_returned'];
        $qd = (int)$ln['qty_damaged'];

        // line pricing: مرتبط بطلب التسليم الأصلي (يشمل الخصم المستخدم)
        $unit = isset($appliedUnit[$iid]) ? (float)$appliedUnit[$iid] : 0.0;

        // ledger movements:
        // sold/returned/damaged all leave OUTLET
        if ($qs>0) ledger_post($pdo, 'OUTLET', (int)$d['outlet_id'], $iid, 0, $qs, 'SETTLEMENT', $settlementId, (int)$me['id'], 'Sold to customers', 'AVAILABLE', 'SOLD_OUT');
        if ($qr>0) {
          ledger_post($pdo, 'OUTLET', (int)$d['outlet_id'], $iid, 0, $qr, 'SETTLEMENT', $settlementId, (int)$me['id'], 'Returned to distributor', 'AVAILABLE', 'RETURN_OUT');
          ledger_post($pdo, 'DISTRIBUTOR', (int)$d['distributor_id'], $iid, $qr, 0, 'SETTLEMENT', $settlementId, (int)$me['id'], 'Return from outlet', 'AVAILABLE', 'RETURN_IN');
        }
        if ($qd>0) {
          ledger_post($pdo, 'OUTLET', (int)$d['outlet_id'], $iid, 0, $qd, 'DAMAGE_PENDING', $settlementId, (int)$me['id'], 'Damaged pending approval', 'AVAILABLE', 'DAMAGED_OUT');
          // ضع الكمية التالفة على عهدة الموزع (قيد الاعتماد)
          ledger_post($pdo, 'DISTRIBUTOR', (int)$d['distributor_id'], $iid, $qd, 0, 'DAMAGE_PENDING', $settlementId, (int)$me['id'], 'Damaged held with distributor pending approval', 'HOLD', 'HOLD_IN');
          // سجل طلب تالف (قيد المراجعة)
          // NOTE: keep insert compatible with older schemas; add richer fields when available.
          ensure_damage_reports_columns($pdo);
          $coverSnap = 0.0;
          try {
            $stC = $pdo->prepare("SELECT cover_price FROM issues WHERE id=?");
            $stC->execute([$iid]);
            $coverSnap = (float)($stC->fetchColumn() ?: 0);
          } catch (Throwable $e) {}

          // Try extended insert (settlement_id/unit/cover) first, fallback to base insert if columns missing.
          try {
            $pdo->prepare("INSERT INTO damage_reports (issue_id, distributor_id, outlet_id, settlement_id, location, qty, unit_price_applied, cover_price, photo_path, status, created_by, created_at)
                           VALUES (?,?,?, ?, 'WITH_OUTLET', ?, ?, ?, ?, 'PENDING', ?, NOW())")
                ->execute([$iid, (int)$d['distributor_id'], (int)$d['outlet_id'], $settlementId, $qd, $unit, $coverSnap, $damagePhotoPath, (int)$me['id']]);
          } catch (Throwable $e) {
            $pdo->prepare("INSERT INTO damage_reports (issue_id, distributor_id, outlet_id, location, qty, photo_path, status, created_by, created_at)
                           VALUES (?,?,?, 'WITH_OUTLET', ?, ?, 'PENDING', ?, NOW())")
                ->execute([$iid, (int)$d['distributor_id'], (int)$d['outlet_id'], $qd, $damagePhotoPath, (int)$me['id']]);
          }
        }

        $gross = round($qs * $unit, 3);
        $comm  = distributor_commission($pdo, (int)$d['distributor_id'], $gross);
        $net   = round(max(0.0, $gross - $comm), 3);

        $pdo->prepare("INSERT INTO outlet_settlement_lines (settlement_id, issue_id, qty_sold, qty_returned, qty_damaged, unit_price_applied, gross_amount, distributor_commission, net_due)
                       VALUES (?,?,?,?,?,?,?,?,?)")
            ->execute([$settlementId, $iid, $qs, $qr, $qd, $unit, $gross, $comm, $net]);

        if ($qs>0) {
          $lineTotal = $gross;
          $total += $lineTotal;
          $pdo->prepare("INSERT INTO invoice_items (invoice_id, issue_id, qty, unit_price, line_total, commission_amount) VALUES (?,?,?,?,?,?)")
              ->execute([$invoiceId, $iid, $qs, $unit, $lineTotal, $comm]);
        }
      }

      $total = round($total, 3);

      // Guard: cash collected cannot exceed outlet due
      if ($cash_collected - $total > 0.0005) {
        throw new RuntimeException('تحصيل نقدي أكبر من المطلوب دفعه');
      }

      if ($cash_collected > 0.0) {
        // يسجل كنقد مع الموزع (HELD) ولا يُقفل الفاتورة حتى يقوم الأدمن بترحيل التحصيل
        $payNo = next_doc_no($pdo, 'PAY', date('Y-m-d'));
        $pdo->prepare("INSERT INTO payments (payment_no, outlet_id, distributor_id, amount, method, paid_at, notes, created_by, status)
                       VALUES (?,?,?,?,'CASH',NOW(),?,?,'HELD')")
            ->execute([$payNo, (int)$d['outlet_id'], (int)$d['distributor_id'], $cash_collected, 'تحصيل تسوية أمانة (معلّق حتى تسليم المكتب)', (int)$me['id']]);
        $payId = (int)$pdo->lastInsertId();
        $alloc = min($cash_collected, $total);
        if ($alloc > 0) {
          $pdo->prepare("INSERT INTO payment_allocations (payment_id, invoice_id, amount_allocated) VALUES (?,?,?)")
              ->execute([$payId, $invoiceId, $alloc]);
        }
      }

      // حدّث إجمالي الفاتورة ثم أعد الاحتساب (يُحسب HELD كمدفوع لأن المنفذ دفع للموزع).
      $pdo->prepare("UPDATE invoices SET total_amount=? WHERE id=?")
          ->execute([$total, $invoiceId]);
      recompute_invoice_totals($pdo, $invoiceId, true);

      audit_log($pdo, (int)$me['id'], 'CREATE', 'outlet_settlements', $settlementId, ['delivery_id'=>$delivery_id,'has_damage'=>$hasDamage]);

      $pdo->commit();
      flash($hasDamage ? 'تم تنفيذ التسوية (التالف بانتظار موافقة الإدارة)' : 'تم تنفيذ التسوية بنجاح');
      redirect('settlements.php');

	    } catch (Throwable $e) {
	      // Avoid secondary exception: "There is no active transaction"
	      if ($pdo->inTransaction()) {
	        $pdo->rollBack();
	      }
	      logger('ERROR', $e->getMessage(), [
	        'type'  => get_class($e),
	        'file'  => $e->getFile(),
	        'line'  => $e->getLine(),
	        'trace' => $e->getTraceAsString(),
	      ]);
	      exit('حدث خطأ غير متوقع. برجاء المحاولة لاحقاً.');
	    }
  }
  if (!$hasDeliveryCol) {
	    echo '<div style="font-family:Tahoma; padding:16px;">'	       .'<h3>مطلوب تحديث قاعدة البيانات</h3>'	       .'<p>شغّل الملف <b>DB_PATCH_SETL.sql</b> مرة واحدة داخل phpMyAdmin ثم أعد فتح الصفحة.</p>'	       .'</div>';
    exit;
  }

}

// Page list
$rows = [];
if ($me['role'] === 'DISTRIBUTOR') {
  if (!$distId) exit('Distributor user not linked');
  // معاملات أمانة تحتاج تسوية: CONSIGNMENT + APPROVED + لا يوجد settlement
  $st = $pdo->prepare("SELECT od.id, od.delivery_no, od.delivery_at, o.name AS outlet_name
                       FROM outlet_deliveries od
                       JOIN outlets o ON o.id=od.outlet_id
                       LEFT JOIN outlet_settlements os ON os.delivery_id = od.id
                       WHERE od.distributor_id=? AND od.deal_type='CONSIGNMENT' AND od.status='APPROVED' AND os.id IS NULL
                       ORDER BY od.id DESC
                       LIMIT 200");
  $st->execute([$distId]);
  $rows = $st->fetchAll();
} else {
  // Admin: التسويات المعلقة (تحتاج اعتماد المدير)
  $st = $pdo->prepare("SELECT
      os.id,
      os.settlement_no,
      os.settlement_at,
      os.status,
      d.name AS distributor_name,
      o.name AS outlet_name,
      od.delivery_no,
      od.delivery_at,
      (SELECT COUNT(*) FROM damage_reports dr WHERE dr.settlement_id=os.id AND dr.status='PENDING') AS pending_damage
    FROM outlet_settlements os
    JOIN distributors d ON d.id=os.distributor_id
    JOIN outlets o ON o.id=os.outlet_id
    LEFT JOIN outlet_deliveries od ON od.id=os.delivery_id
    WHERE os.status='SUBMITTED'
    ORDER BY os.id DESC
    LIMIT 200");
  $st->execute();
  $rows = $st->fetchAll();
}

?><!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>تسويات الأمانة — MGZ</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
<style>
  body{font-family:Tahoma,Arial; background:#f4f7f6; margin:0; padding:16px;}
  .topbar{display:flex; align-items:center; justify-content:space-between; gap:10px; flex-wrap:wrap; margin-bottom:12px;}
  .btn{background:#1a73e8; color:#fff; border:none; padding:10px 14px; border-radius:10px; cursor:pointer; font-size:14px;}
  .btn-secondary{background:#6c757d;}
  .card{background:#fff; border-radius:14px; padding:14px; box-shadow:0 2px 10px rgba(0,0,0,.06);}
  table{width:100%; border-collapse:collapse;}
  th,td{padding:10px; border-bottom:1px solid #eee; font-size:13px; text-align:right;}
  th{background:#fafafa;}
  .muted{color:#666; font-size:12px;}
  .linkbtn{border:none; background:#f1f5ff; padding:8px 10px; border-radius:10px; cursor:pointer;}
  .modal-backdrop{position:fixed; inset:0; background:rgba(0,0,0,.55); display:none; align-items:center; justify-content:center; z-index:9999;}
  .modal{background:#fff; width:min(980px, 96vw); max-height:90vh; overflow:auto; border-radius:16px; padding:14px;}
  @media (max-width:520px){ .modal{width:98vw; height:92vh; max-height:92vh;} .btn{width:100%;} }
  .modal-header{display:flex; justify-content:space-between; align-items:center; gap:10px; margin-bottom:10px;}
  .x{background:transparent; border:none; font-size:22px; cursor:pointer;}
  input{width:100%; padding:10px; border:1px solid #ddd; border-radius:10px; font-size:14px;}
  .num{width:90px; max-width:90px; padding:8px; font-size:13px;}
  .money{white-space:nowrap; direction:ltr; unicode-bidi:embed;}
</style>

<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/ui.css">
<script defer src="../assets/ui.js"></script>
</head>
<body>

<div class="topbar">
  <a href="../dashboard.php" class="btn btn-secondary" style="text-decoration:none;">⬅ العودة</a>
  <div style="flex:1"></div>
</div>

<?php show_flash(); ?>

<div class="card">
  <h3 style="margin:0 0 10px 0;">
    <?= $me['role']==='ADMIN' ? '🧾 تسويات أمانة معلّقة (تحتاج اعتماد)' : '🧾 معاملات أمانة تحتاج تسوية' ?>
  </h3>
  <table>
    <thead>
      <?php if ($me['role']==='ADMIN'): ?>
        <tr>
          <th>رقم التسوية</th>
          <th>الموزع</th>
          <th>المنفذ</th>
          <th>رقم التوريد</th>
          <th>تاريخ التسوية</th>
          <th>تالف معلّق</th>
          <th>إجراء</th>
        </tr>
      <?php else: ?>
        <tr><th>#</th><th>المنفذ</th><th>التاريخ</th><th></th></tr>
      <?php endif; ?>
    </thead>
    <tbody>
      <?php if(!$rows): ?>
        <tr>
          <td colspan="<?= $me['role']==='ADMIN' ? 7 : 4 ?>" class="muted">
            <?= $me['role']==='ADMIN' ? 'لا يوجد تسويات معلّقة' : 'لا يوجد معاملات تحتاج تسوية' ?>
          </td>
        </tr>
      <?php else: foreach($rows as $r): ?>
        <?php if ($me['role']==='ADMIN'): ?>
          <tr>
            <td><?=h($r['settlement_no'] ?: $r['id'])?></td>
            <td><?=h($r['distributor_name'])?></td>
            <td><?=h($r['outlet_name'])?></td>
            <td><?=h($r['delivery_no'] ?: '-')?></td>
            <td><?=h(date('Y-m-d H:i', strtotime($r['settlement_at'])))?></td>
            <td><?= (int)$r['pending_damage'] ?></td>
            <td style="white-space:nowrap;">
              <form method="post" style="display:inline;" onsubmit="return confirm('اعتماد التسوية؟');">
                <?= csrf_field() ?>
                <input type="hidden" name="op" value="settlement_decide">
                <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                <input type="hidden" name="decision" value="APPROVED">
                <button class="btn btn-success btn-sm" <?= ((int)$r['pending_damage']>0)?'disabled':'' ?>>اعتماد</button>
              </form>
              <button class="btn btn-danger btn-sm" data-reject="<?= (int)$r['id'] ?>">رفض</button>
            </td>
          </tr>
        <?php else: ?>
          <tr>
            <td><?=h($r['delivery_no'] ?: $r['id'])?></td>
            <td><?=h($r['outlet_name'])?></td>
            <td><?=h(date('Y-m-d H:i', strtotime($r['delivery_at'])))?></td>
            <td><button class="linkbtn" data-settle="<?= (int)$r['id'] ?>">تسوية</button></td>
          </tr>
        <?php endif; ?>
      <?php endforeach; endif; ?>
    </tbody>
  </table>
</div>

<!-- Settlement Modal -->
<div class="modal-backdrop" id="mSettle">
  <div class="modal" role="dialog" aria-modal="true">
    <div class="modal-header">
      <div>
        <div style="font-weight:700;">تنفيذ تسوية أمانة</div>
        <div class="muted" id="mInfo">...</div>
      </div>
      <button class="x" data-close="#mSettle">×</button>
    </div>

    <form method="post" enctype="multipart/form-data" id="frmSettle">
      <?= csrf_field() ?>
      <input type="hidden" name="form_token" value="<?= h(form_token("settlement")) ?>">
      <input type="hidden" name="op" value="settle_delivery">
      <input type="hidden" name="delivery_id" id="deliveryId">

      <div class="card" style="margin-bottom:10px;">
        <div class="muted">لكل سطر: (تحصيل + مرتجع + تالف) لازم يساوي الكمية الأصلية</div>
      </div>

      <div class="card" style="padding:0; overflow:hidden;">
        <table id="tblLines">
          <thead>
            <tr>
              <th>المنتج</th>
              <th>الإصدار</th>
              <th>الأصلية</th>
              <th>سعر الغلاف</th>
              <th>السعر بعد الخصم</th>
              <th>الخصم</th>
              <th>تحصيل</th>
              <th>مرتجع</th>
              <th>تالف</th>
            </tr>
          </thead>
          <tbody id="linesBody"></tbody>
        </table>
      </div>

      <div style="margin-top:10px;" class="card">
        <label class="muted">تحصيل نقدي</label>
        <input type="number" step="0.001" min="0" name="cash_collected" id="cashCollected" value="0">
        <div class="muted" style="margin-top:6px;">
          المطلوب دفعه من المنفذ: <b id="dueAmount">0</b> ر.ع
          <span id="dueHint"></span>
        </div>
        <div class="muted" style="margin-top:6px;">(التحصيل يتسجل معلّق حتى تسليم المكتب)</div>
      </div>

      <div class="card" id="damageBox" style="margin-top:10px; display:none;">
        <label class="muted">صورة التالف (إجباري لو في تالف)</label>
        <input type="file" name="damage_photo" id="damagePhoto" accept="image/*">
      </div>

      <div style="margin-top:12px; display:flex; gap:10px; flex-wrap:wrap;">
        <button type="submit" class="btn" id="btnDo" disabled>تنفيذ التسوية</button>
        <button type="button" class="btn btn-secondary" data-close="#mSettle">إلغاء</button>
      </div>

    </form>
  </div>
</div>

<script>
// Format OMR without trailing zeros (e.g., 500.000 -> 500, 10.000 -> 10)
function fmtOMR(n){
  const x = Number(n || 0);
  let s = x.toFixed(3);
  s = s.replace(/\.0+$/,'');
  s = s.replace(/(\.[0-9]*?)0+$/,'$1');
  s = s.replace(/\.$/,'');
  return s === '' ? '0' : s;
}
const $ = (s, el=document) => el.querySelector(s);
const $$ = (s, el=document) => Array.from(el.querySelectorAll(s));

function openModal(id){ document.getElementById(id).style.display='flex'; }
function closeModal(sel){ const m=document.querySelector(sel); if(m) m.style.display='none'; }

document.addEventListener('click', (e)=>{ const c=e.target.getAttribute('data-close'); if(c) closeModal(c); });

$$('[data-settle]').forEach(btn=>{
  btn.addEventListener('click', async ()=>{
    const id = btn.getAttribute('data-settle');
    openModal('mSettle');
    $('#linesBody').innerHTML = '<tr><td colspan="9" class="muted">...</td></tr>';
    const r = await fetch('settlements.php?action=api_delivery&id='+encodeURIComponent(id));
    const js = await r.json();
    if(!js.ok){ $('#linesBody').innerHTML = `<tr><td colspan="9" class="muted">${escapeHtml(js.error||'خطأ')}</td></tr>`; return; }
    $('#deliveryId').value = id;
    $('#mInfo').textContent = `المنفذ: ${js.delivery.outlet_name} — التاريخ: ${js.delivery.delivery_at}`;
    renderLines(js.items);
    validate();
  });
});


// Admin reject settlement
$$('[data-reject]').forEach(btn=>{
  btn.addEventListener('click', ()=>{
    const id = btn.getAttribute('data-reject');
    const reason = prompt('سبب الرفض (اختياري):') || '';
    const f = document.createElement('form');
    f.method = 'post';
    f.innerHTML = `<?= csrf_field() ?>
      <input type="hidden" name="op" value="settlement_decide">
      <input type="hidden" name="id" value="${id}">
      <input type="hidden" name="decision" value="REJECTED">
      <input type="hidden" name="reason" value="${reason.replace(/"/g,'&quot;')}">`;
    document.body.appendChild(f);
    f.submit();
  });
});

function renderLines(items){
  const tb = $('#linesBody');
  tb.innerHTML='';
  items.forEach(it=>{
    const m = (it.issue_month!==undefined && it.issue_month!==null) ? String(it.issue_month).padStart(2,'0') : '';
    const y = (it.issue_year!==undefined && it.issue_year!==null) ? String(it.issue_year) : '';
    const label = `عدد ${it.issue_number||''} - ${m}/${y}`;
    const cover = Number(it.cover_price||0);
    const unit = (it.unit_price_applied!==undefined && it.unit_price_applied!==null) ? Number(it.unit_price_applied) : cover;
    const discPer = Math.max(0, cover - unit);
    const discPct = cover > 0 ? (discPer / cover) * 100 : 0;
    const distAvail = Number(it.dist_avail_qty||0);
    const distHold  = Number(it.dist_hold_qty||0);
    const tr = document.createElement('tr');
    tr.dataset.orig = String(it.qty_delivered);
    tr.dataset.cover = String(cover);
    tr.dataset.unit = String(unit);
    tr.dataset.discPer = String(discPer);
    tr.dataset.discPct = String(discPct);
    tr.dataset.distAvail = String(distAvail);
    tr.dataset.distHold  = String(distHold);
    tr.innerHTML = `
      <td>${escapeHtml(it.product_name)}</td>
      <td>
        ${escapeHtml(label)}
        <div class="muted" style="margin-top:4px;">المتاح عندك: <b class="bAvail">${distAvail}</b> — التالف المعلّق: <b class="bHold">${distHold}</b></div>
      </td>
      <td>${it.qty_delivered}</td>
      <td class="money"><b class="bCover">${fmtOMR(cover)}</b> ر.ع</td>
      <td class="money"><b class="bUnit">${fmtOMR(unit)}</b> ر.ع</td>
      <td class="money">
        <b class="bDisc">${fmtOMR(0)}</b> ر.ع
        <div class="muted" style="margin-top:4px;">نسبة الخصم: <b class="bDiscPct">${discPer>0 ? fmtOMR(discPct) : '0'}</b>%</div>
      </td>
      <td>
        <input type="number" min="0" step="1" value="0" name="qty_sold[]" class="qSold num">
        <div class="muted" style="margin-top:4px;">المبلغ: <b class="bLine">0</b> ر.ع</div>
      </td>
      <td>
        <input type="number" min="0" step="1" value="0" name="qty_returned[]" class="qRet num">
        <div class="muted" style="margin-top:4px;">مخزونك بعد المرتجع: <b class="bAfterRet">${distAvail}</b></div>
      </td>
      <td>
        <input type="number" min="0" step="1" value="0" name="qty_damaged[]" class="qDam num">
        <div class="muted" style="margin-top:4px;">إجمالي التالف المعلّق: <b class="bAfterDam">${distHold}</b></div>
      </td>
      <input type="hidden" name="issue_id[]" value="${it.issue_id}">
    `;
    tb.appendChild(tr);
    ['qSold','qRet','qDam'].forEach(cls=>{
      tr.querySelector('.'+cls).addEventListener('input', ()=>{ clampRow(tr); recalcRow(tr); validate(); });
    });
    recalcRow(tr);
  });
  recalcTotals();
}

function recalcRow(tr){
  const unit = Number(tr.dataset.unit||0);
  const discPer = Number(tr.dataset.discPer||0);
  const distAvail = Number(tr.dataset.distAvail||0);
  const distHold  = Number(tr.dataset.distHold||0);
  const vs = Number(tr.querySelector('.qSold').value||0);
  const vr = Number(tr.querySelector('.qRet').value||0);
  const vd = Number(tr.querySelector('.qDam').value||0);
  tr.querySelector('.bLine').textContent = fmtOMR(vs * unit);
  if (tr.querySelector('.bDisc')) tr.querySelector('.bDisc').textContent = fmtOMR(vs * discPer);
  tr.querySelector('.bAfterRet').textContent = String(distAvail + vr);
  tr.querySelector('.bAfterDam').textContent = String(distHold + vd);
}

function clampRow(tr){
  const orig = Number(tr.dataset.orig||0);
  const s = tr.querySelector('.qSold');
  const r = tr.querySelector('.qRet');
  const d = tr.querySelector('.qDam');
  let vs = Math.max(0, Math.floor(Number(s.value||0)));
  let vr = Math.max(0, Math.floor(Number(r.value||0)));
  let vd = Math.max(0, Math.floor(Number(d.value||0)));
  // clamp each to orig
  if (vs>orig) vs=orig;
  if (vr>orig) vr=orig;
  if (vd>orig) vd=orig;
  s.value = String(vs); r.value = String(vr); d.value = String(vd);
}

function validate(){
  let ok = true;
  let hasDamage = false;
  recalcTotals();
  const hint = document.getElementById('dueHint');
  if (hint) hint.textContent = '';
  $$('#linesBody tr').forEach(tr=>{
    const orig = Number(tr.dataset.orig||0);
    const vs = Number(tr.querySelector('.qSold').value||0);
    const vr = Number(tr.querySelector('.qRet').value||0);
    const vd = Number(tr.querySelector('.qDam').value||0);
    if ((vs+vr+vd) !== orig) ok = false;
    if (vd>0) hasDamage = true;
  });

  const box = document.getElementById('damageBox');
  if (hasDamage) {
    box.style.display='block';
    const f = document.getElementById('damagePhoto');
    if (!f.files || f.files.length===0) {
      ok = false;
      if (hint) hint.textContent = ' — صورة التالف مطلوبة';
    }
  } else {
    box.style.display='none';
  }

  document.getElementById('btnDo').disabled = !ok;
}

function recalcTotals(){
  let due = 0;
  $$('#linesBody tr').forEach(tr=>{
    const unit = Number(tr.dataset.unit||0);
    const vs = Number(tr.querySelector('.qSold')?.value||0);
    due += (vs * unit);
  });
  due = Math.max(0, due);
  const dueEl = document.getElementById('dueAmount');
  if (dueEl) dueEl.textContent = fmtOMR(due);

  const cash = document.getElementById('cashCollected');
  if (cash) {
    cash.max = String(fmtOMR(due));
    let v = Number(cash.value||0);
    if (v<0) v=0;
    if (v>due) v=due;
    // show without trailing zeros
    cash.value = fmtOMR(v);
  }
}

document.getElementById('cashCollected')?.addEventListener('input', ()=>{ recalcTotals(); validate(); });

$('#damagePhoto')?.addEventListener('change', ()=>validate());

document.getElementById('frmSettle')?.addEventListener('submit', (e)=>{
  // validation already handled
});

function escapeHtml(s){
  return String(s??'').replace(/[&<>"']/g, c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c]));
}
</script>

</body>
</html>