<?php
// Run from CLI: php scripts/integrity_check.php
require_once __DIR__ . '/../app/db.php';

$pdo = db();

$checks = [
  ['table'=>'invoice_items', 'col'=>'invoice_id', 'ref'=>'invoices', 'refcol'=>'id'],
  ['table'=>'payment_allocations', 'col'=>'invoice_id', 'ref'=>'invoices', 'refcol'=>'id'],
  ['table'=>'payment_allocations', 'col'=>'payment_id', 'ref'=>'payments', 'refcol'=>'id'],
  ['table'=>'outlet_delivery_items', 'col'=>'delivery_id', 'ref'=>'outlet_deliveries', 'refcol'=>'id'],
  ['table'=>'distributor_dispatch_items', 'col'=>'dispatch_id', 'ref'=>'distributor_dispatches', 'refcol'=>'id'],
];

$extraSql = [
  // Duplicate settlement for same delivery
  'dup_settlement' => "SELECT delivery_id, COUNT(*) c FROM outlet_settlements GROUP BY delivery_id HAVING COUNT(*)>1",
  // Duplicate invoice for same source
  'dup_invoice_source' => "SELECT source_type, source_id, COUNT(*) c FROM invoices WHERE source_id IS NOT NULL GROUP BY source_type, source_id HAVING COUNT(*)>1",
  // Payment allocations exceed payment amount
  'payment_over_alloc' => "SELECT p.id, p.payment_no, p.amount, COALESCE(SUM(pa.amount_allocated),0) allocated FROM payments p LEFT JOIN payment_allocations pa ON pa.payment_id=p.id GROUP BY p.id HAVING allocated>p.amount",
  // Invoice allocations exceed total
  'invoice_over_paid' => "SELECT i.id, i.invoice_no, i.total_amount, COALESCE(SUM(pa.amount_allocated),0) allocated FROM invoices i LEFT JOIN payment_allocations pa ON pa.invoice_id=i.id GROUP BY i.id HAVING allocated>i.total_amount",
  // Invoice stored totals mismatch
  'invoice_totals_mismatch' => "SELECT i.id, i.invoice_no, i.total_amount, i.total_paid, i.balance_due, ROUND(i.total_amount-i.total_paid,3) recomputed_balance FROM invoices i WHERE ROUND(i.total_amount-i.total_paid,3)<>ROUND(i.balance_due,3)",
  // Negative balances
  'negative_balance' => "SELECT entity_type, entity_id, issue_id, SUM(CASE WHEN status='POSTED' AND bucket='AVAILABLE' THEN qty_in-qty_out ELSE 0 END) bal FROM inventory_ledger GROUP BY entity_type, entity_id, issue_id HAVING bal<0",
  // Outlet delivery ledger mismatch
  'outlet_delivery_ledger_mismatch' => "SELECT l.ref_id delivery_id, l.issue_id, SUM(CASE WHEN l.entity_type='DISTRIBUTOR' THEN 1 ELSE 0 END) has_dist, SUM(CASE WHEN l.entity_type='OUTLET' THEN 1 ELSE 0 END) has_outlet FROM inventory_ledger l WHERE l.ref_type='OUTLET_DELIVERY' AND l.status='POSTED' GROUP BY l.ref_id,l.issue_id HAVING has_dist=0 OR has_outlet=0",
];

echo "MGZ Integrity Check - ".date('Y-m-d H:i:s').PHP_EOL;
foreach ($checks as $c){
  $sql = "SELECT COUNT(*) AS c FROM {$c['table']} t LEFT JOIN {$c['ref']} r ON r.{$c['refcol']}=t.{$c['col']} WHERE r.{$c['refcol']} IS NULL";
  $n = (int)$pdo->query($sql)->fetchColumn();
  echo "- {$c['table']}.{$c['col']} -> {$c['ref']}.{$c['refcol']} : orphans = $n".PHP_EOL;
}

echo PHP_EOL."Extra checks:".PHP_EOL;
foreach ($extraSql as $name=>$sql) {
  $st = $pdo->query($sql);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
  $count = count($rows);
  echo "- $name: $count".PHP_EOL;
  if ($count && $count <= 10) {
    foreach ($rows as $r) {
      echo '  ' . json_encode($r, JSON_UNESCAPED_UNICODE) . PHP_EOL;
    }
  }
}
